
#line 1 "../gen/tmpl/lib.c"
/*
  gsl_Histogram.c
  Ruby/Numo::GSL - GSL wrapper for Ruby/Numo::NArray

  created on: 2017-03-11
  Copyright (C) 2017 Masahiro Tanaka
*/

#include <ruby.h>
#include <assert.h>
#include "numo/narray.h"
#include "numo/template.h"
#include "../numo_gsl.h"
#line 15 "../gen/tmpl/lib.c"
#include <gsl/gsl_histogram.h>
#include <gsl/gsl_histogram2d.h>

static VALUE mG;



#line 1 "../gen/tmpl/cast_1d_contiguous.c"
static VALUE
cast_1d_contiguous(VALUE v1, VALUE type)
{
    narray_t *na;

    v1 = rb_funcall(type, rb_intern("cast"), 1, v1);
    GetNArray(v1,na);
    if (NA_NDIM(na) != 1) {
        rb_raise(nary_eShapeError, "argument should be 1-dimensional array");
    }
    if (!RTEST(na_check_contiguous(v1))) {
        v1 = na_copy(v1);
    }
    return v1;
}


#line 1 "../gen/tmpl/class.c"
/*
  class definition: Numo::GSL::Histogram
*/

static VALUE cHistogram;

static void
histogram_free(void *ptr)
{
    gsl_histogram_free(ptr);
}

static size_t
histogram_memsize(const void *ptr)
{
    return sizeof(gsl_histogram);
}

static const rb_data_type_t histogram_data_type = {
    "Numo::GSL::Histogram",
    {NULL, histogram_free, histogram_memsize,},
    0, 0, RUBY_TYPED_FREE_IMMEDIATELY|RUBY_TYPED_WB_PROTECTED
};



#line 1 "tmpl/c_sizet_f_field.c"
/*
  @overload n
  @return [Integer]

  returns n field in gsl_histogram struct.
*/
static VALUE
histogram_n(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return SIZET2NUM(w->n);
}


#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload range()
  @return [DFloat]    narray of range field in gsl_histogram.

  returns range field in gsl_histogram struct.
  
*/
static VALUE
histogram_range(VALUE self)
{
    gsl_histogram *w;
    double *d;
    VALUE v;
    size_t n, i, shape[1];

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    
    shape[0] = n = w->n+1;
    

#line 29 "tmpl/c_DFloat_f_field.c"
    v = rb_narray_new(cDF, 1, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->range[i];
    }
    return v;
}


#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload bin()
  @return [DFloat]    narray of bin field in gsl_histogram.

  returns bin field in gsl_histogram struct.
  
*/
static VALUE
histogram_bin(VALUE self)
{
    gsl_histogram *w;
    double *d;
    VALUE v;
    size_t n, i, shape[1];

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    
#line 22 "tmpl/c_DFloat_f_field.c"
    shape[0] = n = w->n;
    

#line 29 "tmpl/c_DFloat_f_field.c"
    v = rb_narray_new(cDF, 1, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->bin[i];
    }
    return v;
}


#line 1 "../gen/tmpl/c_new_sizet.c"

#line 5 "../gen/tmpl/c_new_sizet.c"
/*
  @overload new(n)
  @param  [Integer]  n parameter

  allocate instance of Histogram class.

 This function allocates memory for a histogram with n bins, and
returns a pointer to a newly created gsl_histogram struct.  If
insufficient memory is available a null pointer is returned and the
error handler is invoked with an error code of GSL_ENOMEM. The
bins and ranges are not initialized, and should be prepared using one of
the range-setting functions below in order to make the histogram ready
for use.
 */
static VALUE
histogram_s_new(VALUE self, VALUE v1)
{
    gsl_histogram *w;
    w = gsl_histogram_alloc(NUM2SIZET(v1));
    if (!w) {
        rb_raise(rb_eNoMemError,"fail to allocate struct");
    }
    return TypedData_Wrap_Struct(cHistogram, &histogram_data_type, (void*)w);
}


#line 1 "tmpl/c_self_f_set_ranges.c"
/*
  @overload set_ranges(range[])
  @param  [DFloat]   range[]
  @return [Histogram]  self

  This function sets the ranges of the existing histogram h using
the array range of size size.  The values of the histogram
bins are reset to zero.  The range array should contain the
desired bin limits.  The ranges can be arbitrary, subject to the
restriction that they are monotonically increasing.

The following example shows how to create a histogram with logarithmic
bins with ranges [1,10), [10,100) and [100,1000).

gsl_histogram * h = gsl_histogram_alloc (3);

// bin[0] covers the range 1 <= x < 10 
// bin[1] covers the range 10 <= x < 100 
// bin[2] covers the range 100 <= x < 1000 

double range[4] = [ 1.0, 10.0, 100.0, 1000.0 ];

gsl_histogram_set_ranges (h, range, 4);

Note that the size of the range array should be defined to be one
element bigger than the number of bins.  The additional element is
required for the upper value of the final bin.
*/
static VALUE
histogram_set_ranges(VALUE self, VALUE v1)
{
    gsl_histogram *w;
    double *p1;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    v1 = cast_1d_contiguous(v1, cDF);
    p1 = (double*)na_get_pointer_for_read(v1);
    gsl_histogram_set_ranges(w, p1, RNARRAY_SIZE(v1));
    RB_GC_GUARD(v1);
    return self;
}


#line 1 "../gen/tmpl/c_void_f_double_x2.c"
/*
  @overload set_ranges_uniform(xmin,xmax)
  @param  [Float]  xmin
  @param  [Float]  xmax
  @return [Qnil]

  This function sets the ranges of the existing histogram h to cover
the range xmin to xmax uniformly.  The values of the
histogram bins are reset to zero.  The bin ranges are shown in the table
below,

where d is the bin spacing, d = (xmax-xmin)/n.
*/
static VALUE
histogram_set_ranges_uniform(VALUE self, VALUE v1, VALUE v2)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    gsl_histogram_set_ranges_uniform(w, NUM2DBL(v1), NUM2DBL(v2));
    return Qnil;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload memcpy(src)
  @param  [Histogram] src
  @return [Bool]

  This function copies the histogram src into the pre-existing
histogram dest, making dest into an exact copy of src.
The two histograms must be of the same size.
*/
static VALUE
histogram_memcpy(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram, &histogram_data_type, w1);

    stat = gsl_histogram_memcpy(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "../gen/tmpl/c_self_f_DFloat.c"
static void
iter_histogram_increment(na_loop_t *const lp)
{
    size_t   i;
    char    *p1;
    ssize_t  s1;
    size_t  *idx1;
    double   x;
    gsl_histogram *w = (gsl_histogram*)(lp->opt_ptr);
    

    INIT_COUNTER(lp, i);
    INIT_PTR_IDX(lp, 0, p1, s1, idx1);

    if (idx1) {
        for (; i--;) {
            GET_DATA_INDEX(p1,idx1,double,x);
            gsl_histogram_increment(w,x);
        }
    } else {
        for (; i--;) {
            GET_DATA_STRIDE(p1,s1,double,x);
            gsl_histogram_increment(w,x);
        }
    }
}

/*
  @overload increment(x)
  @param  [DFloat]   x
  @return [Histogram]  self

  This function updates the histogram h by adding one (1.0) to the
bin whose range contains the coordinate x. 

If x lies in the valid range of the histogram then the function
returns zero to indicate success.  If x is less than the lower
limit of the histogram then the function returns GSL_EDOM, and
none of bins are modified.  Similarly, if the value of x is greater
than or equal to the upper limit of the histogram then the function
returns GSL_EDOM, and none of the bins are modified.  The error
handler is not called, however, since it is often necessary to compute
histograms for a small range of a larger dataset, ignoring the values
outside the range of interest.
*/
static VALUE
histogram_increment(VALUE self, VALUE v1)
{
    gsl_histogram *w;
    ndfunc_arg_in_t ain[1] = {{cDF,0}};
    ndfunc_t ndf = {iter_histogram_increment, FULL_LOOP, 1,0, ain,0};

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    na_ndloop3(&ndf, w, 1, v1);
    return self;
}


#line 1 "../gen/tmpl/c_self_f_DFloat_x2.c"
static void
iter_histogram_accumulate(na_loop_t *const lp)
{
    size_t   i;
    char    *p1, *p2;
    ssize_t  s1, s2;
    double   x, y;
    gsl_histogram *w = (gsl_histogram*)(lp->opt_ptr);
    

    INIT_COUNTER(lp, i);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);

    for (; i--;) {
        GET_DATA_STRIDE(p1,s1,double,x);
        GET_DATA_STRIDE(p2,s2,double,y);
        gsl_histogram_accumulate(w,x,y);
    }
}

/*
  @overload accumulate(x,weight)
  @param  [DFloat]   x
  @param  [DFloat]   weight
  @return [Histogram]  self

  This function is similar to gsl_histogram_increment but increases
the value of the appropriate bin in the histogram h by the
floating-point number weight.
*/
static VALUE
histogram_accumulate(VALUE self, VALUE v1, VALUE v2)
{
    gsl_histogram *w;
    ndfunc_arg_in_t ain[2] = {{cDF,0},{cDF,0}};
    ndfunc_t ndf = {iter_histogram_accumulate, STRIDE_LOOP, 2,0, ain,0};

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    na_ndloop3(&ndf, w, 2, v1, v2);
    return self;
}


#line 1 "tmpl/c_DFloat_f_get.c"
static void
iter_histogram_get(na_loop_t *const lp)
{
    size_t   n, i;
    char    *p1, *p2;
    ssize_t  s1, s2;
    ssize_t  x1;
    size_t   y1;
    double   y2;
    gsl_histogram *w = (gsl_histogram*)(lp->opt_ptr);

    INIT_COUNTER(lp, n);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);

    for (i=0; i<n; i++) {
        GET_DATA_STRIDE(p1,s1,ssize_t,x1);
        if (x1 < 0) {
            y1 = x1 + w->n;
        } else {
            y1 = x1;
        }
        y2 = gsl_histogram_get(w,y1);
        SET_DATA_STRIDE(p2,s2,double,y2);
    }
}
#undef FUNC

/*
  @overload get(i)
  @param  [Int32/64]  i
  @return [Histogram]  self

  This function returns the contents of the i-th bin of the histogram
h.  If i lies outside the valid range of indices for the
histogram then the error handler is called with an error code of
GSL_EDOM and the function returns 0.
*/
static VALUE
histogram_get(VALUE self, VALUE v1)
{
    gsl_histogram *w;
    ndfunc_arg_in_t ain[1] = {{cSSZ,0}};
    ndfunc_arg_out_t aout[1] = {{cDF,0}};
    ndfunc_t ndf = {iter_histogram_get, STRIDE_LOOP|NDF_EXTRACT, 1,1, ain,aout};

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return na_ndloop3(&ndf, w, 1, v1);
}


#line 1 "tmpl/c_DFloat_x2_f_get_range.c"


#line 9 "tmpl/c_DFloat_x2_f_get_range.c"
static void
iter_histogram_get_range(na_loop_t *const lp)
{
    size_t   n, i;
    char    *p1, *p2, *p3;
    ssize_t  s1, s2, s3;
    ssize_t  x1;
    size_t   y1;
    double   y2, y3;
    gsl_histogram *w = (gsl_histogram*)(lp->opt_ptr);

    INIT_COUNTER(lp, n);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);
    INIT_PTR(lp, 2, p3, s3);

    for (i=0; i<n; i++) {
        GET_DATA_STRIDE(p1,s1,ssize_t,x1);
        if (x1 < 0) {
            y1 = x1 + w->n;
        } else {
            y1 = x1;
        }
        gsl_histogram_get_range(w,y1,&y2,&y3);
        SET_DATA_STRIDE(p2,s2,double,y2);
        SET_DATA_STRIDE(p3,s3,double,y3);
    }
}

/*
  @overload get_range(i)
  @param  [Int32/64]  i  Int32/64 NArray
  @return [Array]     array of [[DFloat] lower, [DFloat] upper]

  This function finds the upper and lower range limits of the i-th
bin of the histogram h.  If the index i is valid then the
corresponding range limits are stored in lower and upper.
The lower limit is inclusive (i.e. events with this coordinate are
included in the bin) and the upper limit is exclusive (i.e. events with
the coordinate of the upper limit are excluded and fall in the
neighboring higher bin, if it exists).  The function returns 0 to
indicate success.  If i lies outside the valid range of indices for
the histogram then the error handler is called and the function returns
an error code of GSL_EDOM.
*/
static VALUE
histogram_get_range(VALUE self, VALUE v1)
{
    gsl_histogram *w;
    ndfunc_arg_in_t ain[1] = {{cSSZ,0}};
    ndfunc_arg_out_t aout[2] = {{cDF,0},{cDF,0}};
    ndfunc_t ndf = {iter_histogram_get_range, STRIDE_LOOP|NDF_EXTRACT, 1,2, ain,aout};

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return na_ndloop3(&ndf, w, 1, v1);
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload max
  @return [Float]

  These functions return the maximum upper and minimum lower range limits
and the number of bins of the histogram h.  They provide a way of
determining these values without accessing the gsl_histogram
struct directly.
*/
static VALUE
histogram_max(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return DBL2NUM(gsl_histogram_max(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload min
  @return [Float]

  These functions return the maximum upper and minimum lower range limits
and the number of bins of the histogram h.  They provide a way of
determining these values without accessing the gsl_histogram
struct directly.
*/
static VALUE
histogram_min(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return DBL2NUM(gsl_histogram_min(w));
}


#line 1 "../gen/tmpl/c_sizet_f_void.c"
/*
  @overload bins()
  @return [Integer]

  These functions return the maximum upper and minimum lower range limits
and the number of bins of the histogram h.  They provide a way of
determining these values without accessing the gsl_histogram
struct directly.
*/
static VALUE
histogram_bins(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return SIZET2NUM(gsl_histogram_bins(w));
}


#line 1 "../gen/tmpl/c_void_f_void.c"
/*
  @overload reset

  This function resets all the bins in the histogram h to zero.
*/
static VALUE
histogram_reset(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);
    gsl_histogram_reset(w);
    return Qnil;
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload max_val
  @return [Float]

  This function returns the maximum value contained in the histogram bins.
*/
static VALUE
histogram_max_val(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return DBL2NUM(gsl_histogram_max_val(w));
}


#line 1 "../gen/tmpl/c_sizet_f_void.c"
/*
  @overload max_bin()
  @return [Integer]

  This function returns the index of the bin containing the maximum
value. In the case where several bins contain the same maximum value the
smallest index is returned.
*/
static VALUE
histogram_max_bin(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return SIZET2NUM(gsl_histogram_max_bin(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload min_val
  @return [Float]

  This function returns the minimum value contained in the histogram bins.
*/
static VALUE
histogram_min_val(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return DBL2NUM(gsl_histogram_min_val(w));
}


#line 1 "../gen/tmpl/c_sizet_f_void.c"
/*
  @overload min_bin()
  @return [Integer]

  This function returns the index of the bin containing the minimum
value. In the case where several bins contain the same maximum value the
smallest index is returned.
*/
static VALUE
histogram_min_bin(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return SIZET2NUM(gsl_histogram_min_bin(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload mean
  @return [Float]

  This function returns the mean of the histogrammed variable, where the
histogram is regarded as a probability distribution. Negative bin values
are ignored for the purposes of this calculation.  The accuracy of the
result is limited by the bin width.
*/
static VALUE
histogram_mean(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return DBL2NUM(gsl_histogram_mean(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload sigma
  @return [Float]

  This function returns the standard deviation of the histogrammed
variable, where the histogram is regarded as a probability
distribution. Negative bin values are ignored for the purposes of this
calculation. The accuracy of the result is limited by the bin width.
*/
static VALUE
histogram_sigma(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return DBL2NUM(gsl_histogram_sigma(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload sum
  @return [Float]

  This function returns the sum of all bin values. Negative bin values
are included in the sum.
*/
static VALUE
histogram_sum(VALUE self)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    return DBL2NUM(gsl_histogram_sum(w));
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload equal_bins_p(h2)
  @param  [Histogram] h2
  @return [Bool]

  This function returns 1 if the all of the individual bin
ranges of the two histograms are identical, and 0
otherwise.
*/
static VALUE
histogram_equal_bins_p(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram, &histogram_data_type, w1);

    stat = gsl_histogram_equal_bins_p(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload add(h2)
  @param  [Histogram] h2
  @return [Bool]

  This function adds the contents of the bins in histogram h2 to the
corresponding bins of histogram h1,  i.e. h'_1(i) = h_1(i) +
h_2(i).  The two histograms must have identical bin ranges.
*/
static VALUE
histogram_add(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram, &histogram_data_type, w1);

    stat = gsl_histogram_add(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload sub(h2)
  @param  [Histogram] h2
  @return [Bool]

  This function subtracts the contents of the bins in histogram h2
from the corresponding bins of histogram h1, i.e. h'_1(i) =
h_1(i) - h_2(i).  The two histograms must have identical bin ranges.
*/
static VALUE
histogram_sub(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram, &histogram_data_type, w1);

    stat = gsl_histogram_sub(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload mul(h2)
  @param  [Histogram] h2
  @return [Bool]

  This function multiplies the contents of the bins of histogram h1
by the contents of the corresponding bins in histogram h2,
i.e. h'_1(i) = h_1(i) * h_2(i).  The two histograms must have
identical bin ranges.
*/
static VALUE
histogram_mul(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram, &histogram_data_type, w1);

    stat = gsl_histogram_mul(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload div(h2)
  @param  [Histogram] h2
  @return [Bool]

  This function divides the contents of the bins of histogram h1 by
the contents of the corresponding bins in histogram h2,
i.e. h'_1(i) = h_1(i) / h_2(i).  The two histograms must have
identical bin ranges.
*/
static VALUE
histogram_div(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram, &histogram_data_type, w1);

    stat = gsl_histogram_div(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "../gen/tmpl/c_void_f_double.c"
/*
  @overload scale(scale)
  @param  [Float]  scale
  @return [Qnil]

  This function multiplies the contents of the bins of histogram h
by the constant scale, i.e. $h'_1(i) = h_1(i) * \hbox{\it scale}$
h'_1(i) = h_1(i) * scale.
*/
static VALUE
histogram_scale(VALUE self, VALUE v1)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    gsl_histogram_scale(w, NUM2DBL(v1));
    return Qnil;
}


#line 1 "../gen/tmpl/c_void_f_double.c"
/*
  @overload shift(offset)
  @param  [Float]  offset
  @return [Qnil]

  This function shifts the contents of the bins of histogram h by
the constant offset, i.e. $h'_1(i) = h_1(i) + \hbox{\it offset}$
h'_1(i) = h_1(i) + offset.
*/
static VALUE
histogram_shift(VALUE self, VALUE v1)
{
    gsl_histogram *w;

    TypedData_Get_Struct(self, gsl_histogram, &histogram_data_type, w);

    gsl_histogram_shift(w, NUM2DBL(v1));
    return Qnil;
}


#line 1 "../gen/tmpl/class.c"
/*
  class definition: Numo::GSL::HistogramPdf
*/

static VALUE cHistogramPdf;

static void
histogram_pdf_free(void *ptr)
{
    gsl_histogram_pdf_free(ptr);
}

static size_t
histogram_pdf_memsize(const void *ptr)
{
    return sizeof(gsl_histogram_pdf);
}

static const rb_data_type_t histogram_pdf_data_type = {
    "Numo::GSL::HistogramPdf",
    {NULL, histogram_pdf_free, histogram_pdf_memsize,},
    0, 0, RUBY_TYPED_FREE_IMMEDIATELY|RUBY_TYPED_WB_PROTECTED
};



#line 1 "tmpl/c_sizet_f_field.c"
/*
  @overload n
  @return [Integer]

  returns n field in gsl_histogram_pdf struct.
*/
static VALUE
histogram_pdf_n(VALUE self)
{
    gsl_histogram_pdf *w;

    TypedData_Get_Struct(self, gsl_histogram_pdf, &histogram_pdf_data_type, w);

    return SIZET2NUM(w->n);
}


#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload range()
  @return [DFloat]    narray of range field in gsl_histogram_pdf.

  returns range field in gsl_histogram_pdf struct.
  
*/
static VALUE
histogram_pdf_range(VALUE self)
{
    gsl_histogram_pdf *w;
    double *d;
    VALUE v;
    size_t n, i, shape[1];

    TypedData_Get_Struct(self, gsl_histogram_pdf, &histogram_pdf_data_type, w);

    
    shape[0] = n = w->n+1;
    

#line 29 "tmpl/c_DFloat_f_field.c"
    v = rb_narray_new(cDF, 1, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->range[i];
    }
    return v;
}


#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload sum()
  @return [DFloat]    narray of sum field in gsl_histogram_pdf.

  returns sum field in gsl_histogram_pdf struct.
  
*/
static VALUE
histogram_pdf_sum(VALUE self)
{
    gsl_histogram_pdf *w;
    double *d;
    VALUE v;
    size_t n, i, shape[1];

    TypedData_Get_Struct(self, gsl_histogram_pdf, &histogram_pdf_data_type, w);

    
#line 22 "tmpl/c_DFloat_f_field.c"
    shape[0] = n = w->n;
    

#line 29 "tmpl/c_DFloat_f_field.c"
    v = rb_narray_new(cDF, 1, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->sum[i];
    }
    return v;
}


#line 1 "../gen/tmpl/c_new_sizet.c"

#line 5 "../gen/tmpl/c_new_sizet.c"
/*
  @overload new(n)
  @param  [Integer]  n parameter

  allocate instance of HistogramPdf class.

 This function allocates memory for a probability distribution with
n bins and returns a pointer to a newly initialized
gsl_histogram_pdf struct. If insufficient memory is available a
null pointer is returned and the error handler is invoked with an error
code of GSL_ENOMEM.
 */
static VALUE
histogram_pdf_s_new(VALUE self, VALUE v1)
{
    gsl_histogram_pdf *w;
    w = gsl_histogram_pdf_alloc(NUM2SIZET(v1));
    if (!w) {
        rb_raise(rb_eNoMemError,"fail to allocate struct");
    }
    return TypedData_Wrap_Struct(cHistogramPdf, &histogram_pdf_data_type, (void*)w);
}


#line 1 "tmpl/c_bool_f_pdf_init.c"

#line 5 "tmpl/c_bool_f_pdf_init.c"
/*
  @overload init(h)
  @param  [HistogramPdf] h
  @return [Bool]

  This function initializes the probability distribution p with
the contents of the histogram h. If any of the bins of h are
negative then the error handler is invoked with an error code of
GSL_EDOM because a probability distribution cannot contain
negative values.
*/
static VALUE
histogram_pdf_init(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram_pdf *w;
    gsl_histogram *w1;

    TypedData_Get_Struct(self, gsl_histogram_pdf, &histogram_pdf_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram, &histogram_data_type, w1);

    stat = gsl_histogram_pdf_init(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "../gen/tmpl/c_DFloat_f_DFloat.c"
static void
iter_histogram_pdf_sample(na_loop_t *const lp)
{
    size_t   i;
    char    *p1, *p2;
    ssize_t  s1, s2;
    double   x, y;
    gsl_histogram_pdf *w = (gsl_histogram_pdf*)(lp->opt_ptr);
    

    INIT_COUNTER(lp, i);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);

    for (; i--; ) {
        GET_DATA_STRIDE(p1,s1,double,x);
        y = gsl_histogram_pdf_sample(w,x);
        SET_DATA_STRIDE(p2,s2,double,y);
    }
}

/*
  @overload sample(r)
  @param  [DFloat]    r
  @return [DFloat]    result

  This function uses r, a uniform random number between zero and
one, to compute a single random sample from the probability distribution
p.  The algorithm used to compute the sample s is given by
the following formula,

s = range[i] + delta * (range[i+1] - range[i])

where i is the index which satisfies 
$sum[i] \le  r < sum[i+1]$
sum[i] <=  r < sum[i+1] and 
delta is 
$(r - sum[i])/(sum[i+1] - sum[i])$
(r - sum[i])/(sum[i+1] - sum[i]).
*/
static VALUE
histogram_pdf_sample(VALUE self, VALUE v1)
{
    gsl_histogram_pdf *w;
    ndfunc_arg_in_t ain[1] = {{cDF,0}};
    ndfunc_arg_out_t aout[1] = {{cDF,0}};
    ndfunc_t ndf = {iter_histogram_pdf_sample, STRIDE_LOOP|NDF_EXTRACT, 1,1, ain,aout};

    TypedData_Get_Struct(self, gsl_histogram_pdf, &histogram_pdf_data_type, w);

    return na_ndloop3(&ndf, w, 1, v1);
}


#line 1 "../gen/tmpl/class.c"
/*
  class definition: Numo::GSL::Histogram2D
*/

static VALUE cHistogram2D;

static void
histogram2d_free(void *ptr)
{
    gsl_histogram2d_free(ptr);
}

static size_t
histogram2d_memsize(const void *ptr)
{
    return sizeof(gsl_histogram2d);
}

static const rb_data_type_t histogram2d_data_type = {
    "Numo::GSL::Histogram2D",
    {NULL, histogram2d_free, histogram2d_memsize,},
    0, 0, RUBY_TYPED_FREE_IMMEDIATELY|RUBY_TYPED_WB_PROTECTED
};



#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload xrange()
  @return [DFloat]    narray of xrange field in gsl_histogram2d.

  returns xrange field in gsl_histogram2d struct.
  
*/
static VALUE
histogram2d_xrange(VALUE self)
{
    gsl_histogram2d *w;
    double *d;
    VALUE v;
    size_t n, i, shape[2];

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    
    shape[0] = n = w->nx+1;
    

#line 29 "tmpl/c_DFloat_f_field.c"
    v = rb_narray_new(cDF, 2, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->xrange[i];
    }
    return v;
}


#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload yrange()
  @return [DFloat]    narray of yrange field in gsl_histogram2d.

  returns yrange field in gsl_histogram2d struct.
  
*/
static VALUE
histogram2d_yrange(VALUE self)
{
    gsl_histogram2d *w;
    double *d;
    VALUE v;
    size_t n, i, shape[2];

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    
    shape[0] = n = w->ny+1;
    

#line 29 "tmpl/c_DFloat_f_field.c"
    v = rb_narray_new(cDF, 2, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->yrange[i];
    }
    return v;
}


#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload bin()
  @return [DFloat]    narray of bin field in gsl_histogram2d.

  returns bin field in gsl_histogram2d struct.
  
*/
static VALUE
histogram2d_bin(VALUE self)
{
    gsl_histogram2d *w;
    double *d;
    VALUE v;
    size_t n, i, shape[2];

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    
#line 24 "tmpl/c_DFloat_f_field.c"
    n = w->nx * w->ny;
    shape[0] = w->nx;
    shape[1] = w->ny;
    

    v = rb_narray_new(cDF, 2, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->bin[i];
    }
    return v;
}


#line 1 "../gen/tmpl/c_new_sizet_x2.c"

#line 5 "../gen/tmpl/c_new_sizet_x2.c"
/*
  @overload new(nx,ny)
  @param  [Integer]  nx parameter
  @param  [Integer]  ny parameter

  allocate instance of Histogram2D class.

This function allocates memory for a two-dimensional histogram with
nx bins in the x direction and ny bins in the y direction.
The function returns a pointer to a newly created gsl_histogram2d
struct. If insufficient memory is available a null pointer is returned
and the error handler is invoked with an error code of
GSL_ENOMEM. The bins and ranges must be initialized with one of
the functions below before the histogram is ready for use. */
static VALUE
histogram2d_s_new(VALUE klass, VALUE v1, VALUE v2)
{
    gsl_histogram2d *w;
    w = gsl_histogram2d_alloc(NUM2SIZET(v1), NUM2SIZET(v2));
    if (!w) {
        rb_raise(rb_eNoMemError,"fail to allocate struct");
    }
    return TypedData_Wrap_Struct(cHistogram2D, &histogram2d_data_type, (void*)w);
}


#line 1 "tmpl/c_self_f_2d_set_ranges.c"
/*
  @overload set_ranges(xrange[],yrange[])
  @param  [DFloat]   xrange[]
  @param  [DFloat]   yrange[]
  @return [Histogram2D]  self

  This function sets the ranges of the existing histogram h using
the arrays xrange and yrange of size xsize and
ysize respectively.  The values of the histogram bins are reset to
zero.
*/
static VALUE
histogram2d_set_ranges(VALUE self, VALUE v1, VALUE v2)
{
    gsl_histogram2d *w;
    double *p1, *p2;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    v1 = cast_1d_contiguous(v1, cDF);
    p1 = (double*)na_get_pointer_for_read(v1);
    v2 = cast_1d_contiguous(v2, cDF);
    p2 = (double*)na_get_pointer_for_read(v2);

    gsl_histogram2d_set_ranges(w, p1, RNARRAY_SIZE(v1), p2, RNARRAY_SIZE(v2));
    RB_GC_GUARD(v1);
    RB_GC_GUARD(v2);
    return self;
}


#line 1 "../gen/tmpl/c_void_f_double_x4.c"
/*
  @overload set_ranges_uniform(xmin,xmax,ymin,ymax)
  @param  [Float]  xmin
  @param  [Float]  xmax
  @param  [Float]  ymin
  @param  [Float]  ymax
  @return [Qnil]

  This function sets the ranges of the existing histogram h to cover
the ranges xmin to xmax and ymin to ymax
uniformly.  The values of the histogram bins are reset to zero.
*/
static VALUE
histogram2d_set_ranges_uniform(VALUE self, VALUE v1, VALUE v2, VALUE v3, VALUE v4)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    gsl_histogram2d_set_ranges_uniform(w, NUM2DBL(v1),NUM2DBL(v2),NUM2DBL(v3),NUM2DBL(v4));
    return Qnil;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload memcpy(src)
  @param  [Histogram2D] src
  @return [Bool]

  This function copies the histogram src into the pre-existing
histogram dest, making dest into an exact copy of src.
The two histograms must be of the same size.
*/
static VALUE
histogram2d_memcpy(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram2d *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram2d, &histogram2d_data_type, w1);

    stat = gsl_histogram2d_memcpy(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "../gen/tmpl/c_self_f_DFloat_x2.c"
static void
iter_histogram2d_increment(na_loop_t *const lp)
{
    size_t   i;
    char    *p1, *p2;
    ssize_t  s1, s2;
    double   x, y;
    gsl_histogram2d *w = (gsl_histogram2d*)(lp->opt_ptr);
    

    INIT_COUNTER(lp, i);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);

    for (; i--;) {
        GET_DATA_STRIDE(p1,s1,double,x);
        GET_DATA_STRIDE(p2,s2,double,y);
        gsl_histogram2d_increment(w,x,y);
    }
}

/*
  @overload increment(x,y)
  @param  [DFloat]   x
  @param  [DFloat]   y
  @return [Histogram2D]  self

  This function updates the histogram h by adding one (1.0) to the
bin whose x and y ranges contain the coordinates (x,y).

If the point (x,y) lies inside the valid ranges of the
histogram then the function returns zero to indicate success.  If
(x,y) lies outside the limits of the histogram then the
function returns GSL_EDOM, and none of the bins are modified.  The
error handler is not called, since it is often necessary to compute
histograms for a small range of a larger dataset, ignoring any
coordinates outside the range of interest.
*/
static VALUE
histogram2d_increment(VALUE self, VALUE v1, VALUE v2)
{
    gsl_histogram2d *w;
    ndfunc_arg_in_t ain[2] = {{cDF,0},{cDF,0}};
    ndfunc_t ndf = {iter_histogram2d_increment, STRIDE_LOOP, 2,0, ain,0};

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    na_ndloop3(&ndf, w, 2, v1, v2);
    return self;
}


#line 1 "../gen/tmpl/c_self_f_DFloat_x3.c"
static void
iter_histogram2d_accumulate(na_loop_t *const lp)
{
    size_t   i;
    char    *p1, *p2, *p3;
    ssize_t  s1, s2, s3;
    double   x1, x2, x3;
    gsl_histogram2d *w = (gsl_histogram2d*)(lp->opt_ptr);
    

    INIT_COUNTER(lp, i);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);
    INIT_PTR(lp, 2, p3, s3);

    for (; i--;) {
        GET_DATA_STRIDE(p1,s1,double,x1);
        GET_DATA_STRIDE(p2,s2,double,x2);
        GET_DATA_STRIDE(p3,s3,double,x3);
        gsl_histogram2d_accumulate(w,x1,x2,x3);
    }
}

/*
  @overload accumulate(x,y,weight)
  @param  [DFloat]   x
  @param  [DFloat]   y
  @param  [DFloat]   weight
  @return [Histogram2D]  self

  This function is similar to gsl_histogram2d_increment but increases
the value of the appropriate bin in the histogram h by the
floating-point number weight.
*/
static VALUE
histogram2d_accumulate(VALUE self, VALUE v1, VALUE v2, VALUE v3)
{
    gsl_histogram2d *w;
    ndfunc_arg_in_t ain[3] = {{cDF,0},{cDF,0},{cDF,0}};
    ndfunc_t ndf = {iter_histogram2d_accumulate, STRIDE_LOOP, 3,0, ain,0};

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    na_ndloop3(&ndf, w, 3, v1, v2, v3);
    return self;
}


#line 1 "tmpl/c_DFloat_f_2d_get.c"
static void
iter_histogram2d_get(na_loop_t *const lp)
{
    size_t   n, i;
    char    *p1, *p2, *p3;
    ssize_t  s1, s2, s3;
    ssize_t  x1, x2;
    size_t   y1, y2;
    double   y3;
    gsl_histogram2d *w = (gsl_histogram2d*)(lp->opt_ptr);

    INIT_COUNTER(lp, n);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);
    INIT_PTR(lp, 2, p3, s3);

    for (i=0; i<n; i++) {
        GET_DATA_STRIDE(p1,s1,ssize_t,x1);
        GET_DATA_STRIDE(p2,s2,ssize_t,x2);
        if (x1 < 0) {
            y1 = x1 + w->nx;
        } else {
            y1 = x1;
        }
        if (x2 < 0) {
            y2 = x2 + w->ny;
        } else {
            y2 = x2;
        }
        y3 = gsl_histogram2d_get(w,y1,y2);
        SET_DATA_STRIDE(p3,s3,double,y3);
    }
}

/*
  @overload get(i, j)
  @param  [Int32/64]  i
  @param  [Int32/64]  j
  @return [Histogram2D]  self

  This function returns the contents of the (i,j)-th bin of the
histogram h.  If (i,j) lies outside the valid range of
indices for the histogram then the error handler is called with an error
code of GSL_EDOM and the function returns 0.
*/
static VALUE
histogram2d_get(VALUE self, VALUE v1, VALUE v2)
{
    gsl_histogram2d *w;
    ndfunc_arg_in_t ain[2] = {{cSSZ,0},{cSSZ,0}};
    ndfunc_arg_out_t aout[1] = {{cDF,0}};
    ndfunc_t ndf = {iter_histogram2d_get, STRIDE_LOOP|NDF_EXTRACT, 2,1, ain,aout};

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return na_ndloop3(&ndf, w, 2, v1, v2);
}


#line 1 "tmpl/c_DFloat_x2_f_get_range.c"


#line 9 "tmpl/c_DFloat_x2_f_get_range.c"
static void
iter_histogram2d_get_xrange(na_loop_t *const lp)
{
    size_t   n, i;
    char    *p1, *p2, *p3;
    ssize_t  s1, s2, s3;
    ssize_t  x1;
    size_t   y1;
    double   y2, y3;
    gsl_histogram2d *w = (gsl_histogram2d*)(lp->opt_ptr);

    INIT_COUNTER(lp, n);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);
    INIT_PTR(lp, 2, p3, s3);

    for (i=0; i<n; i++) {
        GET_DATA_STRIDE(p1,s1,ssize_t,x1);
        if (x1 < 0) {
            y1 = x1 + w->nx;
        } else {
            y1 = x1;
        }
        gsl_histogram2d_get_xrange(w,y1,&y2,&y3);
        SET_DATA_STRIDE(p2,s2,double,y2);
        SET_DATA_STRIDE(p3,s3,double,y3);
    }
}

/*
  @overload get_xrange(i)
  @param  [Int32/64]  i  Int32/64 NArray
  @return [Array]     array of [[DFloat] xlower, [DFloat] xupper]

  These functions find the upper and lower range limits of the i-th
and j-th bins in the x and y directions of the histogram h.
The range limits are stored in xlower and xupper or
ylower and yupper.  The lower limits are inclusive
(i.e. events with these coordinates are included in the bin) and the
upper limits are exclusive (i.e. events with the value of the upper
limit are not included and fall in the neighboring higher bin, if it
exists).  The functions return 0 to indicate success.  If i or
j lies outside the valid range of indices for the histogram then
the error handler is called with an error code of GSL_EDOM.
*/
static VALUE
histogram2d_get_xrange(VALUE self, VALUE v1)
{
    gsl_histogram2d *w;
    ndfunc_arg_in_t ain[1] = {{cSSZ,0}};
    ndfunc_arg_out_t aout[2] = {{cDF,0},{cDF,0}};
    ndfunc_t ndf = {iter_histogram2d_get_xrange, STRIDE_LOOP|NDF_EXTRACT, 1,2, ain,aout};

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return na_ndloop3(&ndf, w, 1, v1);
}


#line 1 "tmpl/c_DFloat_x2_f_get_range.c"


#line 9 "tmpl/c_DFloat_x2_f_get_range.c"
static void
iter_histogram2d_get_yrange(na_loop_t *const lp)
{
    size_t   n, i;
    char    *p1, *p2, *p3;
    ssize_t  s1, s2, s3;
    ssize_t  x1;
    size_t   y1;
    double   y2, y3;
    gsl_histogram2d *w = (gsl_histogram2d*)(lp->opt_ptr);

    INIT_COUNTER(lp, n);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);
    INIT_PTR(lp, 2, p3, s3);

    for (i=0; i<n; i++) {
        GET_DATA_STRIDE(p1,s1,ssize_t,x1);
        if (x1 < 0) {
            y1 = x1 + w->ny;
        } else {
            y1 = x1;
        }
        gsl_histogram2d_get_yrange(w,y1,&y2,&y3);
        SET_DATA_STRIDE(p2,s2,double,y2);
        SET_DATA_STRIDE(p3,s3,double,y3);
    }
}

/*
  @overload get_yrange(j)
  @param  [Int32/64]  j  Int32/64 NArray
  @return [Array]     array of [[DFloat] ylower, [DFloat] yupper]

  These functions find the upper and lower range limits of the i-th
and j-th bins in the x and y directions of the histogram h.
The range limits are stored in xlower and xupper or
ylower and yupper.  The lower limits are inclusive
(i.e. events with these coordinates are included in the bin) and the
upper limits are exclusive (i.e. events with the value of the upper
limit are not included and fall in the neighboring higher bin, if it
exists).  The functions return 0 to indicate success.  If i or
j lies outside the valid range of indices for the histogram then
the error handler is called with an error code of GSL_EDOM.
*/
static VALUE
histogram2d_get_yrange(VALUE self, VALUE v1)
{
    gsl_histogram2d *w;
    ndfunc_arg_in_t ain[1] = {{cSSZ,0}};
    ndfunc_arg_out_t aout[2] = {{cDF,0},{cDF,0}};
    ndfunc_t ndf = {iter_histogram2d_get_yrange, STRIDE_LOOP|NDF_EXTRACT, 1,2, ain,aout};

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return na_ndloop3(&ndf, w, 1, v1);
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload xmax
  @return [Float]

  These functions return the maximum upper and minimum lower range limits
and the number of bins for the x and y directions of the histogram
h.  They provide a way of determining these values without
accessing the gsl_histogram2d struct directly.
*/
static VALUE
histogram2d_xmax(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_xmax(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload xmin
  @return [Float]

  These functions return the maximum upper and minimum lower range limits
and the number of bins for the x and y directions of the histogram
h.  They provide a way of determining these values without
accessing the gsl_histogram2d struct directly.
*/
static VALUE
histogram2d_xmin(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_xmin(w));
}


#line 1 "../gen/tmpl/c_sizet_f_void.c"
/*
  @overload nx()
  @return [Integer]

  These functions return the maximum upper and minimum lower range limits
and the number of bins for the x and y directions of the histogram
h.  They provide a way of determining these values without
accessing the gsl_histogram2d struct directly.
*/
static VALUE
histogram2d_nx(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return SIZET2NUM(gsl_histogram2d_nx(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload ymax
  @return [Float]

  These functions return the maximum upper and minimum lower range limits
and the number of bins for the x and y directions of the histogram
h.  They provide a way of determining these values without
accessing the gsl_histogram2d struct directly.
*/
static VALUE
histogram2d_ymax(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_ymax(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload ymin
  @return [Float]

  These functions return the maximum upper and minimum lower range limits
and the number of bins for the x and y directions of the histogram
h.  They provide a way of determining these values without
accessing the gsl_histogram2d struct directly.
*/
static VALUE
histogram2d_ymin(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_ymin(w));
}


#line 1 "../gen/tmpl/c_sizet_f_void.c"
/*
  @overload ny()
  @return [Integer]

  These functions return the maximum upper and minimum lower range limits
and the number of bins for the x and y directions of the histogram
h.  They provide a way of determining these values without
accessing the gsl_histogram2d struct directly.
*/
static VALUE
histogram2d_ny(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return SIZET2NUM(gsl_histogram2d_ny(w));
}


#line 1 "../gen/tmpl/c_void_f_void.c"
/*
  @overload reset

  This function resets all the bins of the histogram h to zero.
*/
static VALUE
histogram2d_reset(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);
    gsl_histogram2d_reset(w);
    return Qnil;
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload max_val
  @return [Float]

  This function returns the maximum value contained in the histogram bins.
*/
static VALUE
histogram2d_max_val(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_max_val(w));
}


#line 1 "../gen/tmpl/c_sizet_x2_f_void.c"
/*
  @overload max_bin()
  @return [Array]   array of [h,i]

  This function finds the indices of the bin containing the maximum value
in the histogram h and stores the result in (i,j). In
the case where several bins contain the same maximum value the first bin
found is returned.
*/
static VALUE
histogram2d_max_bin(VALUE self)
{
    size_t i, j;
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    gsl_histogram2d_max_bin(w, &i, &j);
    return rb_assoc_new(SIZET2NUM(i), SIZET2NUM(j));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload min_val
  @return [Float]

  This function returns the minimum value contained in the histogram bins.
*/
static VALUE
histogram2d_min_val(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_min_val(w));
}


#line 1 "../gen/tmpl/c_sizet_x2_f_void.c"
/*
  @overload min_bin()
  @return [Array]   array of [h,i]

  This function finds the indices of the bin containing the minimum value
in the histogram h and stores the result in (i,j). In
the case where several bins contain the same maximum value the first bin
found is returned.
*/
static VALUE
histogram2d_min_bin(VALUE self)
{
    size_t i, j;
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    gsl_histogram2d_min_bin(w, &i, &j);
    return rb_assoc_new(SIZET2NUM(i), SIZET2NUM(j));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload xmean
  @return [Float]

  This function returns the mean of the histogrammed x variable, where the
histogram is regarded as a probability distribution. Negative bin values
are ignored for the purposes of this calculation.
*/
static VALUE
histogram2d_xmean(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_xmean(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload ymean
  @return [Float]

  This function returns the mean of the histogrammed y variable, where the
histogram is regarded as a probability distribution. Negative bin values
are ignored for the purposes of this calculation.
*/
static VALUE
histogram2d_ymean(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_ymean(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload xsigma
  @return [Float]

  This function returns the standard deviation of the histogrammed
x variable, where the histogram is regarded as a probability
distribution. Negative bin values are ignored for the purposes of this
calculation.
*/
static VALUE
histogram2d_xsigma(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_xsigma(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload ysigma
  @return [Float]

  This function returns the standard deviation of the histogrammed
y variable, where the histogram is regarded as a probability
distribution. Negative bin values are ignored for the purposes of this
calculation.
*/
static VALUE
histogram2d_ysigma(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_ysigma(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload cov
  @return [Float]

  This function returns the covariance of the histogrammed x and y
variables, where the histogram is regarded as a probability
distribution. Negative bin values are ignored for the purposes of this
calculation.
*/
static VALUE
histogram2d_cov(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_cov(w));
}


#line 1 "../gen/tmpl/c_double_f_void.c"
/*
  @overload sum
  @return [Float]

  This function returns the sum of all bin values. Negative bin values
are included in the sum.
*/
static VALUE
histogram2d_sum(VALUE self)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    return DBL2NUM(gsl_histogram2d_sum(w));
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload equal_bins_p(h2)
  @param  [Histogram2D] h2
  @return [Bool]

  This function returns 1 if all the individual bin ranges of the two
histograms are identical, and 0 otherwise.
*/
static VALUE
histogram2d_equal_bins_p(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram2d *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram2d, &histogram2d_data_type, w1);

    stat = gsl_histogram2d_equal_bins_p(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload add(h2)
  @param  [Histogram2D] h2
  @return [Bool]

  This function adds the contents of the bins in histogram h2 to the
corresponding bins of histogram h1,
i.e. h'_1(i,j) = h_1(i,j) + h_2(i,j).
The two histograms must have identical bin ranges.
*/
static VALUE
histogram2d_add(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram2d *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram2d, &histogram2d_data_type, w1);

    stat = gsl_histogram2d_add(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload sub(h2)
  @param  [Histogram2D] h2
  @return [Bool]

  This function subtracts the contents of the bins in histogram h2 from the
corresponding bins of histogram h1,
i.e. h'_1(i,j) = h_1(i,j) - h_2(i,j).
The two histograms must have identical bin ranges.
*/
static VALUE
histogram2d_sub(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram2d *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram2d, &histogram2d_data_type, w1);

    stat = gsl_histogram2d_sub(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload mul(h2)
  @param  [Histogram2D] h2
  @return [Bool]

  This function multiplies the contents of the bins of histogram h1
by the contents of the corresponding bins in histogram h2,
i.e. h'_1(i,j) = h_1(i,j) * h_2(i,j).
The two histograms must have identical bin ranges.
*/
static VALUE
histogram2d_mul(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram2d *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram2d, &histogram2d_data_type, w1);

    stat = gsl_histogram2d_mul(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "tmpl/c_bool_f_other.c"
/*
  @overload div(h2)
  @param  [Histogram2D] h2
  @return [Bool]

  This function divides the contents of the bins of histogram h1
by the contents of the corresponding bins in histogram h2,
i.e. h'_1(i,j) = h_1(i,j) / h_2(i,j).
The two histograms must have identical bin ranges.
*/
static VALUE
histogram2d_div(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram2d *w, *w1;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram2d, &histogram2d_data_type, w1);

    stat = gsl_histogram2d_div(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "../gen/tmpl/c_void_f_double.c"
/*
  @overload scale(scale)
  @param  [Float]  scale
  @return [Qnil]

  This function multiplies the contents of the bins of histogram h
by the constant scale, i.e. $h'_1(i,j) = h_1(i,j) * \hbox{\it scale}$
h'_1(i,j) = h_1(i,j) scale.
*/
static VALUE
histogram2d_scale(VALUE self, VALUE v1)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    gsl_histogram2d_scale(w, NUM2DBL(v1));
    return Qnil;
}


#line 1 "../gen/tmpl/c_void_f_double.c"
/*
  @overload shift(offset)
  @param  [Float]  offset
  @return [Qnil]

  This function shifts the contents of the bins of histogram h
by the constant offset, i.e. $h'_1(i,j) = h_1(i,j) + \hbox{\it offset}$
h'_1(i,j) = h_1(i,j) + offset.
*/
static VALUE
histogram2d_shift(VALUE self, VALUE v1)
{
    gsl_histogram2d *w;

    TypedData_Get_Struct(self, gsl_histogram2d, &histogram2d_data_type, w);

    gsl_histogram2d_shift(w, NUM2DBL(v1));
    return Qnil;
}


#line 1 "../gen/tmpl/class.c"
/*
  class definition: Numo::GSL::Histogram2DPdf
*/

static VALUE cHistogram2DPdf;

static void
histogram2d_pdf_free(void *ptr)
{
    gsl_histogram2d_pdf_free(ptr);
}

static size_t
histogram2d_pdf_memsize(const void *ptr)
{
    return sizeof(gsl_histogram2d_pdf);
}

static const rb_data_type_t histogram2d_pdf_data_type = {
    "Numo::GSL::Histogram2DPdf",
    {NULL, histogram2d_pdf_free, histogram2d_pdf_memsize,},
    0, 0, RUBY_TYPED_FREE_IMMEDIATELY|RUBY_TYPED_WB_PROTECTED
};



#line 1 "tmpl/c_sizet_f_field.c"
/*
  @overload nx
  @return [Integer]

  returns nx field in gsl_histogram2d_pdf struct.
*/
static VALUE
histogram2d_pdf_nx(VALUE self)
{
    gsl_histogram2d_pdf *w;

    TypedData_Get_Struct(self, gsl_histogram2d_pdf, &histogram2d_pdf_data_type, w);

    return SIZET2NUM(w->nx);
}


#line 1 "tmpl/c_sizet_f_field.c"
/*
  @overload ny
  @return [Integer]

  returns ny field in gsl_histogram2d_pdf struct.
*/
static VALUE
histogram2d_pdf_ny(VALUE self)
{
    gsl_histogram2d_pdf *w;

    TypedData_Get_Struct(self, gsl_histogram2d_pdf, &histogram2d_pdf_data_type, w);

    return SIZET2NUM(w->ny);
}


#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload xrange()
  @return [DFloat]    narray of xrange field in gsl_histogram2d_pdf.

  returns xrange field in gsl_histogram2d_pdf struct.
  
*/
static VALUE
histogram2d_pdf_xrange(VALUE self)
{
    gsl_histogram2d_pdf *w;
    double *d;
    VALUE v;
    size_t n, i, shape[2];

    TypedData_Get_Struct(self, gsl_histogram2d_pdf, &histogram2d_pdf_data_type, w);

    
    shape[0] = n = w->nx+1;
    

#line 29 "tmpl/c_DFloat_f_field.c"
    v = rb_narray_new(cDF, 2, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->xrange[i];
    }
    return v;
}


#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload yrange()
  @return [DFloat]    narray of yrange field in gsl_histogram2d_pdf.

  returns yrange field in gsl_histogram2d_pdf struct.
  
*/
static VALUE
histogram2d_pdf_yrange(VALUE self)
{
    gsl_histogram2d_pdf *w;
    double *d;
    VALUE v;
    size_t n, i, shape[2];

    TypedData_Get_Struct(self, gsl_histogram2d_pdf, &histogram2d_pdf_data_type, w);

    
    shape[0] = n = w->ny+1;
    

#line 29 "tmpl/c_DFloat_f_field.c"
    v = rb_narray_new(cDF, 2, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->yrange[i];
    }
    return v;
}


#line 1 "tmpl/c_DFloat_f_field.c"
#line 2 "tmpl/c_DFloat_f_field.c"
/*
  @overload sum()
  @return [DFloat]    narray of sum field in gsl_histogram2d_pdf.

  returns sum field in gsl_histogram2d_pdf struct.
  
*/
static VALUE
histogram2d_pdf_sum(VALUE self)
{
    gsl_histogram2d_pdf *w;
    double *d;
    VALUE v;
    size_t n, i, shape[2];

    TypedData_Get_Struct(self, gsl_histogram2d_pdf, &histogram2d_pdf_data_type, w);

    
#line 24 "tmpl/c_DFloat_f_field.c"
    n = w->nx * w->ny;
    shape[0] = w->nx;
    shape[1] = w->ny;
    

    v = rb_narray_new(cDF, 2, shape);
    d = (double*)na_get_pointer_for_write(v);
    for (i=0; i<n; i++) {
        d[i] = w->sum[i];
    }
    return v;
}


#line 1 "../gen/tmpl/c_new_sizet_x2.c"

#line 5 "../gen/tmpl/c_new_sizet_x2.c"
/*
  @overload new(nx,ny)
  @param  [Integer]  nx parameter
  @param  [Integer]  ny parameter

  allocate instance of Histogram2DPdf class.

This function allocates memory for a two-dimensional probability
distribution of size nx-by-ny and returns a pointer to a
newly initialized gsl_histogram2d_pdf struct. If insufficient
memory is available a null pointer is returned and the error handler is
invoked with an error code of GSL_ENOMEM. */
static VALUE
histogram2d_pdf_s_new(VALUE klass, VALUE v1, VALUE v2)
{
    gsl_histogram2d_pdf *w;
    w = gsl_histogram2d_pdf_alloc(NUM2SIZET(v1), NUM2SIZET(v2));
    if (!w) {
        rb_raise(rb_eNoMemError,"fail to allocate struct");
    }
    return TypedData_Wrap_Struct(cHistogram2DPdf, &histogram2d_pdf_data_type, (void*)w);
}


#line 1 "tmpl/c_bool_f_pdf_init.c"

#line 5 "tmpl/c_bool_f_pdf_init.c"
/*
  @overload init(h)
  @param  [Histogram2DPdf] h
  @return [Bool]

  This function initializes the two-dimensional probability distribution
calculated p from the histogram h.  If any of the bins of
h are negative then the error handler is invoked with an error
code of GSL_EDOM because a probability distribution cannot
contain negative values.
*/
static VALUE
histogram2d_pdf_init(VALUE self, VALUE v1)
{
    int stat;
    gsl_histogram2d_pdf *w;
    gsl_histogram2d *w1;

    TypedData_Get_Struct(self, gsl_histogram2d_pdf, &histogram2d_pdf_data_type, w);
    TypedData_Get_Struct(v1, gsl_histogram2d, &histogram2d_data_type, w1);

    stat = gsl_histogram2d_pdf_init(w, w1);
    return (stat) ? Qtrue: Qfalse;
}


#line 1 "../gen/tmpl/c_DFloat_x2_f_DFloat_x2.c"
static void
iter_histogram2d_pdf_sample(na_loop_t *const lp)
{
    size_t   i;
    char    *p1, *p2, *p3, *p4;
    ssize_t  s1, s2, s3, s4;
    double   x1, x2, y3, y4;
    gsl_histogram2d_pdf *w = (gsl_histogram2d_pdf*)(lp->opt_ptr);
    

    INIT_COUNTER(lp, i);
    INIT_PTR(lp, 0, p1, s1);
    INIT_PTR(lp, 1, p2, s2);
    INIT_PTR(lp, 2, p3, s3);
    INIT_PTR(lp, 3, p4, s4);

    for (; i--; ) {
        GET_DATA_STRIDE(p1,s1,double,x1);
        GET_DATA_STRIDE(p2,s2,double,x2);
        gsl_histogram2d_pdf_sample(w,x1,x2,&y3,&y4);
        SET_DATA_STRIDE(p3,s3,double,y3);
        SET_DATA_STRIDE(p4,s4,double,y4);
    }
}

/*
  @overload sample(r1,r2)
  @param  [DFloat]   r1
  @param  [DFloat]   r2
  @return [Array]    array of [[DFloat] x, [DFloat] y]

  This function uses two uniform random numbers between zero and one,
r1 and r2, to compute a single random sample from the
two-dimensional probability distribution p.
*/
static VALUE
histogram2d_pdf_sample(VALUE self, VALUE v1, VALUE v2)
{
    gsl_histogram2d_pdf *w;
    ndfunc_arg_in_t ain[2] = {{cDF,0},{cDF,0}};
    ndfunc_arg_out_t aout[2] = {{cDF,0},{cDF,0}};
    ndfunc_t ndf = {iter_histogram2d_pdf_sample, STRIDE_LOOP|NDF_EXTRACT, 2,2, ain,aout};

    TypedData_Get_Struct(self, gsl_histogram2d_pdf, &histogram2d_pdf_data_type, w);

    return na_ndloop3(&ndf, w, 2, v1, v2);
}


#line 28 "../gen/tmpl/lib.c"
void
Init_histogram(void)
{
    VALUE mN;
    mN = rb_define_module("Numo");
    mG = rb_define_module_under(mN, "GSL");

    


#line 1 "../gen/tmpl/init_class.c"

    /*
      Document-class: Numo::GSL::Histogram
      
    */
    {
    cHistogram = rb_define_class_under(mG, "Histogram", rb_cObject);
    
    rb_undef_alloc_func(cHistogram);
    rb_define_method(cHistogram, "n", histogram_n, 0);
    rb_define_method(cHistogram, "range", histogram_range, 0);
    rb_define_method(cHistogram, "bin", histogram_bin, 0);
    rb_define_singleton_method(cHistogram, "new", histogram_s_new, 1);
    rb_define_method(cHistogram, "set_ranges", histogram_set_ranges, 1);
    rb_define_method(cHistogram, "set_ranges_uniform", histogram_set_ranges_uniform, 2);
    rb_define_method(cHistogram, "memcpy", histogram_memcpy, 1);
    rb_define_method(cHistogram, "increment", histogram_increment, 1);
    rb_define_method(cHistogram, "accumulate", histogram_accumulate, 2);
    rb_define_method(cHistogram, "get", histogram_get, 1);
    rb_define_method(cHistogram, "get_range", histogram_get_range, 1);
    rb_define_method(cHistogram, "max", histogram_max, 0);
    rb_define_method(cHistogram, "min", histogram_min, 0);
    rb_define_method(cHistogram, "bins", histogram_bins, 0);
    rb_define_method(cHistogram, "reset", histogram_reset, 0);
    rb_define_method(cHistogram, "max_val", histogram_max_val, 0);
    rb_define_method(cHistogram, "max_bin", histogram_max_bin, 0);
    rb_define_method(cHistogram, "min_val", histogram_min_val, 0);
    rb_define_method(cHistogram, "min_bin", histogram_min_bin, 0);
    rb_define_method(cHistogram, "mean", histogram_mean, 0);
    rb_define_method(cHistogram, "sigma", histogram_sigma, 0);
    rb_define_method(cHistogram, "sum", histogram_sum, 0);
    rb_define_method(cHistogram, "equal_bins_p", histogram_equal_bins_p, 1);
    rb_define_method(cHistogram, "add", histogram_add, 1);
    rb_define_method(cHistogram, "sub", histogram_sub, 1);
    rb_define_method(cHistogram, "mul", histogram_mul, 1);
    rb_define_method(cHistogram, "div", histogram_div, 1);
    rb_define_method(cHistogram, "scale", histogram_scale, 1);
    rb_define_method(cHistogram, "shift", histogram_shift, 1);
#line 10 "../gen/tmpl/init_class.c"
    }

#line 1 "../gen/tmpl/init_class.c"

    /*
      Document-class: Numo::GSL::HistogramPdf
      
    */
    {
    cHistogramPdf = rb_define_class_under(mG, "HistogramPdf", rb_cObject);
    
    rb_undef_alloc_func(cHistogramPdf);
    rb_define_method(cHistogramPdf, "n", histogram_pdf_n, 0);
    rb_define_method(cHistogramPdf, "range", histogram_pdf_range, 0);
    rb_define_method(cHistogramPdf, "sum", histogram_pdf_sum, 0);
    rb_define_singleton_method(cHistogramPdf, "new", histogram_pdf_s_new, 1);
    rb_define_method(cHistogramPdf, "init", histogram_pdf_init, 1);
    rb_define_method(cHistogramPdf, "sample", histogram_pdf_sample, 1);
#line 10 "../gen/tmpl/init_class.c"
    }

#line 1 "../gen/tmpl/init_class.c"

    /*
      Document-class: Numo::GSL::Histogram2D
      
    */
    {
    cHistogram2D = rb_define_class_under(mG, "Histogram2D", rb_cObject);
    
    rb_undef_alloc_func(cHistogram2D);
    rb_define_method(cHistogram2D, "xrange", histogram2d_xrange, 0);
    rb_define_method(cHistogram2D, "yrange", histogram2d_yrange, 0);
    rb_define_method(cHistogram2D, "bin", histogram2d_bin, 0);
    rb_define_singleton_method(cHistogram2D, "new", histogram2d_s_new, 2);
    rb_define_method(cHistogram2D, "set_ranges", histogram2d_set_ranges, 2);
    rb_define_method(cHistogram2D, "set_ranges_uniform", histogram2d_set_ranges_uniform, 4);
    rb_define_method(cHistogram2D, "memcpy", histogram2d_memcpy, 1);
    rb_define_method(cHistogram2D, "increment", histogram2d_increment, 2);
    rb_define_method(cHistogram2D, "accumulate", histogram2d_accumulate, 3);
    rb_define_method(cHistogram2D, "get", histogram2d_get, 2);
    rb_define_method(cHistogram2D, "get_xrange", histogram2d_get_xrange, 1);
    rb_define_method(cHistogram2D, "get_yrange", histogram2d_get_yrange, 1);
    rb_define_method(cHistogram2D, "xmax", histogram2d_xmax, 0);
    rb_define_method(cHistogram2D, "xmin", histogram2d_xmin, 0);
    rb_define_method(cHistogram2D, "nx", histogram2d_nx, 0);
    rb_define_method(cHistogram2D, "ymax", histogram2d_ymax, 0);
    rb_define_method(cHistogram2D, "ymin", histogram2d_ymin, 0);
    rb_define_method(cHistogram2D, "ny", histogram2d_ny, 0);
    rb_define_method(cHistogram2D, "reset", histogram2d_reset, 0);
    rb_define_method(cHistogram2D, "max_val", histogram2d_max_val, 0);
    rb_define_method(cHistogram2D, "max_bin", histogram2d_max_bin, 0);
    rb_define_method(cHistogram2D, "min_val", histogram2d_min_val, 0);
    rb_define_method(cHistogram2D, "min_bin", histogram2d_min_bin, 0);
    rb_define_method(cHistogram2D, "xmean", histogram2d_xmean, 0);
    rb_define_method(cHistogram2D, "ymean", histogram2d_ymean, 0);
    rb_define_method(cHistogram2D, "xsigma", histogram2d_xsigma, 0);
    rb_define_method(cHistogram2D, "ysigma", histogram2d_ysigma, 0);
    rb_define_method(cHistogram2D, "cov", histogram2d_cov, 0);
    rb_define_method(cHistogram2D, "sum", histogram2d_sum, 0);
    rb_define_method(cHistogram2D, "equal_bins_p", histogram2d_equal_bins_p, 1);
    rb_define_method(cHistogram2D, "add", histogram2d_add, 1);
    rb_define_method(cHistogram2D, "sub", histogram2d_sub, 1);
    rb_define_method(cHistogram2D, "mul", histogram2d_mul, 1);
    rb_define_method(cHistogram2D, "div", histogram2d_div, 1);
    rb_define_method(cHistogram2D, "scale", histogram2d_scale, 1);
    rb_define_method(cHistogram2D, "shift", histogram2d_shift, 1);
#line 10 "../gen/tmpl/init_class.c"
    }

#line 1 "../gen/tmpl/init_class.c"

    /*
      Document-class: Numo::GSL::Histogram2DPdf
      
    */
    {
    cHistogram2DPdf = rb_define_class_under(mG, "Histogram2DPdf", rb_cObject);
    
    rb_undef_alloc_func(cHistogram2DPdf);
    rb_define_method(cHistogram2DPdf, "nx", histogram2d_pdf_nx, 0);
    rb_define_method(cHistogram2DPdf, "ny", histogram2d_pdf_ny, 0);
    rb_define_method(cHistogram2DPdf, "xrange", histogram2d_pdf_xrange, 0);
    rb_define_method(cHistogram2DPdf, "yrange", histogram2d_pdf_yrange, 0);
    rb_define_method(cHistogram2DPdf, "sum", histogram2d_pdf_sum, 0);
    rb_define_singleton_method(cHistogram2DPdf, "new", histogram2d_pdf_s_new, 2);
    rb_define_method(cHistogram2DPdf, "init", histogram2d_pdf_init, 1);
    rb_define_method(cHistogram2DPdf, "sample", histogram2d_pdf_sample, 2);
#line 10 "../gen/tmpl/init_class.c"
    }
#line 41 "../gen/tmpl/lib.c"
}
