/**
 * Capabilities supported by the agent.
 *
 * Advertised during initialization to inform the client about
 * available features and content types.
 *
 * See protocol docs: [Agent Capabilities](https://agentclientprotocol.com/protocol/initialization#agent-capabilities)
 */
export type AgentCapabilities = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Whether the agent supports `session/load`.
     */
    loadSession?: boolean;
    /**
     * MCP capabilities supported by the agent.
     */
    mcpCapabilities?: McpCapabilities;
    /**
     * Prompt capabilities supported by the agent.
     */
    promptCapabilities?: PromptCapabilities;
    sessionCapabilities?: SessionCapabilities;
};
export type AgentNotification = {
    method: string;
    params?: SessionNotification | ExtNotification | null;
};
export type AgentRequest = {
    id: RequestId;
    method: string;
    params?: WriteTextFileRequest | ReadTextFileRequest | RequestPermissionRequest | CreateTerminalRequest | TerminalOutputRequest | ReleaseTerminalRequest | WaitForTerminalExitRequest | KillTerminalCommandRequest | ExtRequest | null;
};
export type AgentResponse = {
    id: RequestId;
    /**
     * All possible responses that an agent can send to a client.
     *
     * This enum is used internally for routing RPC responses. You typically won't need
     * to use this directly - the responses are handled automatically by the connection.
     *
     * These are responses to the corresponding `ClientRequest` variants.
     */
    result: InitializeResponse | AuthenticateResponse | NewSessionResponse | LoadSessionResponse | ListSessionsResponse | ForkSessionResponse | ResumeSessionResponse | SetSessionModeResponse | SetSessionConfigOptionResponse | PromptResponse | SetSessionModelResponse | ExtResponse;
} | {
    error: _Error;
    id: RequestId;
};
/**
 * Optional annotations for the client. The client can use annotations to inform how objects are used or displayed
 */
export type Annotations = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    audience?: Array<Role> | null;
    lastModified?: string | null;
    priority?: number | null;
};
/**
 * Audio provided to or from an LLM.
 */
export type AudioContent = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    annotations?: Annotations | null;
    data: string;
    mimeType: string;
};
/**
 * Describes an available authentication method.
 */
export type AuthMethod = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Optional description providing more details about this authentication method.
     */
    description?: string | null;
    /**
     * Unique identifier for this authentication method.
     */
    id: string;
    /**
     * Human-readable name of the authentication method.
     */
    name: string;
};
/**
 * Request parameters for the authenticate method.
 *
 * Specifies which authentication method to use.
 */
export type AuthenticateRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The ID of the authentication method to use.
     * Must be one of the methods advertised in the initialize response.
     */
    methodId: string;
};
/**
 * Response to the `authenticate` method.
 */
export type AuthenticateResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
};
/**
 * Information about a command.
 */
export type AvailableCommand = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Human-readable description of what the command does.
     */
    description: string;
    /**
     * Input for the command if required
     */
    input?: AvailableCommandInput | null;
    /**
     * Command name (e.g., `create_plan`, `research_codebase`).
     */
    name: string;
};
/**
 * All text that was typed after the command name is provided as input.
 */
export type AvailableCommandInput = UnstructuredCommandInput;
/**
 * Available commands are ready or have changed
 */
export type AvailableCommandsUpdate = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Commands the agent can execute
     */
    availableCommands: Array<AvailableCommand>;
};
/**
 * Binary resource contents.
 */
export type BlobResourceContents = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    blob: string;
    mimeType?: string | null;
    uri: string;
};
/**
 * Notification to cancel ongoing operations for a session.
 *
 * See protocol docs: [Cancellation](https://agentclientprotocol.com/protocol/prompt-turn#cancellation)
 */
export type CancelNotification = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The ID of the session to cancel operations for.
     */
    sessionId: SessionId;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Notification to cancel an ongoing request.
 *
 * See protocol docs: [Cancellation](https://agentclientprotocol.com/protocol/cancellation)
 *
 * @experimental
 */
export type CancelRequestNotification = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The ID of the request to cancel.
     */
    requestId: RequestId;
};
/**
 * Capabilities supported by the client.
 *
 * Advertised during initialization to inform the agent about
 * available features and methods.
 *
 * See protocol docs: [Client Capabilities](https://agentclientprotocol.com/protocol/initialization#client-capabilities)
 */
export type ClientCapabilities = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * File system capabilities supported by the client.
     * Determines which file operations the agent can request.
     */
    fs?: FileSystemCapability;
    /**
     * Whether the Client support all `terminal*` methods.
     */
    terminal?: boolean;
};
export type ClientNotification = {
    method: string;
    params?: CancelNotification | ExtNotification | null;
};
export type ClientRequest = {
    id: RequestId;
    method: string;
    params?: InitializeRequest | AuthenticateRequest | NewSessionRequest | LoadSessionRequest | ListSessionsRequest | ForkSessionRequest | ResumeSessionRequest | SetSessionModeRequest | SetSessionConfigOptionRequest | PromptRequest | SetSessionModelRequest | ExtRequest | null;
};
export type ClientResponse = {
    id: RequestId;
    /**
     * All possible responses that a client can send to an agent.
     *
     * This enum is used internally for routing RPC responses. You typically won't need
     * to use this directly - the responses are handled automatically by the connection.
     *
     * These are responses to the corresponding `AgentRequest` variants.
     */
    result: WriteTextFileResponse | ReadTextFileResponse | RequestPermissionResponse | CreateTerminalResponse | TerminalOutputResponse | ReleaseTerminalResponse | WaitForTerminalExitResponse | KillTerminalCommandResponse | ExtResponse;
} | {
    error: _Error;
    id: RequestId;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Session configuration options have been updated.
 *
 * @experimental
 */
export type ConfigOptionUpdate = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The full set of configuration options and their current values.
     */
    configOptions: Array<SessionConfigOption>;
};
/**
 * Standard content block (text, images, resources).
 */
export type Content = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The actual content block.
     */
    content: ContentBlock;
};
/**
 * Content blocks represent displayable information in the Agent Client Protocol.
 *
 * They provide a structured way to handle various types of user-facing content—whether
 * it's text from language models, images for analysis, or embedded resources for context.
 *
 * Content blocks appear in:
 * - User prompts sent via `session/prompt`
 * - Language model output streamed through `session/update` notifications
 * - Progress updates and results from tool calls
 *
 * This structure is compatible with the Model Context Protocol (MCP), enabling
 * agents to seamlessly forward content from MCP tool outputs without transformation.
 *
 * See protocol docs: [Content](https://agentclientprotocol.com/protocol/content)
 */
export type ContentBlock = (TextContent & {
    type: "text";
}) | (ImageContent & {
    type: "image";
}) | (AudioContent & {
    type: "audio";
}) | (ResourceLink & {
    type: "resource_link";
}) | (EmbeddedResource & {
    type: "resource";
});
/**
 * A streamed item of content
 */
export type ContentChunk = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * A single item of content
     */
    content: ContentBlock;
};
/**
 * Request to create a new terminal and execute a command.
 */
export type CreateTerminalRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Array of command arguments.
     */
    args?: Array<string>;
    /**
     * The command to execute.
     */
    command: string;
    /**
     * Working directory for the command (absolute path).
     */
    cwd?: string | null;
    /**
     * Environment variables for the command.
     */
    env?: Array<EnvVariable>;
    /**
     * Maximum number of output bytes to retain.
     *
     * When the limit is exceeded, the Client truncates from the beginning of the output
     * to stay within the limit.
     *
     * The Client MUST ensure truncation happens at a character boundary to maintain valid
     * string output, even if this means the retained output is slightly less than the
     * specified limit.
     */
    outputByteLimit?: number | null;
    /**
     * The session ID for this request.
     */
    sessionId: SessionId;
};
/**
 * Response containing the ID of the created terminal.
 */
export type CreateTerminalResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The unique identifier for the created terminal.
     */
    terminalId: string;
};
/**
 * The current mode of the session has changed
 *
 * See protocol docs: [Session Modes](https://agentclientprotocol.com/protocol/session-modes)
 */
export type CurrentModeUpdate = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The ID of the current mode
     */
    currentModeId: SessionModeId;
};
/**
 * A diff representing file modifications.
 *
 * Shows changes to files in a format suitable for display in the client UI.
 *
 * See protocol docs: [Content](https://agentclientprotocol.com/protocol/tool-calls#content)
 */
export type Diff = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The new content after modification.
     */
    newText: string;
    /**
     * The original content (None for new files).
     */
    oldText?: string | null;
    /**
     * The file path being modified.
     */
    path: string;
};
/**
 * The contents of a resource, embedded into a prompt or tool call result.
 */
export type EmbeddedResource = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    annotations?: Annotations | null;
    resource: EmbeddedResourceResource;
};
/**
 * Resource content that can be embedded in a message.
 */
export type EmbeddedResourceResource = TextResourceContents | BlobResourceContents;
/**
 * An environment variable to set when launching an MCP server.
 */
export type EnvVariable = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The name of the environment variable.
     */
    name: string;
    /**
     * The value to set for the environment variable.
     */
    value: string;
};
/**
 * JSON-RPC error object.
 *
 * Represents an error that occurred during method execution, following the
 * JSON-RPC 2.0 error object specification with optional additional data.
 *
 * See protocol docs: [JSON-RPC Error Object](https://www.jsonrpc.org/specification#error_object)
 */
export type _Error = {
    /**
     * A number indicating the error type that occurred.
     * This must be an integer as defined in the JSON-RPC specification.
     */
    code: ErrorCode;
    /**
     * Optional primitive or structured value that contains additional information about the error.
     * This may include debugging information or context-specific details.
     */
    data?: unknown;
    /**
     * A string providing a short description of the error.
     * The message should be limited to a concise single sentence.
     */
    message: string;
};
/**
 * Predefined error codes for common JSON-RPC and ACP-specific errors.
 *
 * These codes follow the JSON-RPC 2.0 specification for standard errors
 * and use the reserved range (-32000 to -32099) for protocol-specific errors.
 */
export type ErrorCode = -32700 | -32600 | -32601 | -32602 | -32603 | -32800 | -32000 | -32002 | number;
/**
 * Allows the Agent to send an arbitrary notification that is not part of the ACP spec.
 * Extension notifications provide a way to send one-way messages for custom functionality
 * while maintaining protocol compatibility.
 *
 * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
 */
export type ExtNotification = unknown;
/**
 * Allows for sending an arbitrary request that is not part of the ACP spec.
 * Extension methods provide a way to add custom functionality while maintaining
 * protocol compatibility.
 *
 * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
 */
export type ExtRequest = unknown;
/**
 * Allows for sending an arbitrary response to an [`ExtRequest`] that is not part of the ACP spec.
 * Extension methods provide a way to add custom functionality while maintaining
 * protocol compatibility.
 *
 * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
 */
export type ExtResponse = unknown;
/**
 * Filesystem capabilities supported by the client.
 * File system capabilities that a client may support.
 *
 * See protocol docs: [FileSystem](https://agentclientprotocol.com/protocol/initialization#filesystem)
 */
export type FileSystemCapability = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Whether the Client supports `fs/read_text_file` requests.
     */
    readTextFile?: boolean;
    /**
     * Whether the Client supports `fs/write_text_file` requests.
     */
    writeTextFile?: boolean;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Request parameters for forking an existing session.
 *
 * Creates a new session based on the context of an existing one, allowing
 * operations like generating summaries without affecting the original session's history.
 *
 * Only available if the Agent supports the `session.fork` capability.
 *
 * @experimental
 */
export type ForkSessionRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The working directory for this session.
     */
    cwd: string;
    /**
     * List of MCP servers to connect to for this session.
     */
    mcpServers?: Array<McpServer>;
    /**
     * The ID of the session to fork.
     */
    sessionId: SessionId;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Response from forking an existing session.
 *
 * @experimental
 */
export type ForkSessionResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Initial session configuration options if supported by the Agent.
     *
     * @experimental
     */
    configOptions?: Array<SessionConfigOption> | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Initial model state if supported by the Agent
     *
     * @experimental
     */
    models?: SessionModelState | null;
    /**
     * Initial mode state if supported by the Agent
     *
     * See protocol docs: [Session Modes](https://agentclientprotocol.com/protocol/session-modes)
     */
    modes?: SessionModeState | null;
    /**
     * Unique identifier for the newly created forked session.
     */
    sessionId: SessionId;
};
/**
 * An HTTP header to set when making requests to the MCP server.
 */
export type HttpHeader = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The name of the HTTP header.
     */
    name: string;
    /**
     * The value to set for the HTTP header.
     */
    value: string;
};
/**
 * An image provided to or from an LLM.
 */
export type ImageContent = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    annotations?: Annotations | null;
    data: string;
    mimeType: string;
    uri?: string | null;
};
/**
 * Metadata about the implementation of the client or agent.
 * Describes the name and version of an MCP implementation, with an optional
 * title for UI representation.
 */
export type Implementation = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Intended for programmatic or logical use, but can be used as a display
     * name fallback if title isn’t present.
     */
    name: string;
    /**
     * Intended for UI and end-user contexts — optimized to be human-readable
     * and easily understood.
     *
     * If not provided, the name should be used for display.
     */
    title?: string | null;
    /**
     * Version of the implementation. Can be displayed to the user or used
     * for debugging or metrics purposes. (e.g. "1.0.0").
     */
    version: string;
};
/**
 * Request parameters for the initialize method.
 *
 * Sent by the client to establish connection and negotiate capabilities.
 *
 * See protocol docs: [Initialization](https://agentclientprotocol.com/protocol/initialization)
 */
export type InitializeRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Capabilities supported by the client.
     */
    clientCapabilities?: ClientCapabilities;
    /**
     * Information about the Client name and version sent to the Agent.
     *
     * Note: in future versions of the protocol, this will be required.
     */
    clientInfo?: Implementation | null;
    /**
     * The latest protocol version supported by the client.
     */
    protocolVersion: ProtocolVersion;
};
/**
 * Response to the `initialize` method.
 *
 * Contains the negotiated protocol version and agent capabilities.
 *
 * See protocol docs: [Initialization](https://agentclientprotocol.com/protocol/initialization)
 */
export type InitializeResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Capabilities supported by the agent.
     */
    agentCapabilities?: AgentCapabilities;
    /**
     * Information about the Agent name and version sent to the Client.
     *
     * Note: in future versions of the protocol, this will be required.
     */
    agentInfo?: Implementation | null;
    /**
     * Authentication methods supported by the agent.
     */
    authMethods?: Array<AuthMethod>;
    /**
     * The protocol version the client specified if supported by the agent,
     * or the latest protocol version supported by the agent.
     *
     * The client should disconnect, if it doesn't support this version.
     */
    protocolVersion: ProtocolVersion;
};
/**
 * Request to kill a terminal command without releasing the terminal.
 */
export type KillTerminalCommandRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The session ID for this request.
     */
    sessionId: SessionId;
    /**
     * The ID of the terminal to kill.
     */
    terminalId: string;
};
/**
 * Response to terminal/kill command method
 */
export type KillTerminalCommandResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Request parameters for listing existing sessions.
 *
 * Only available if the Agent supports the `listSessions` capability.
 *
 * @experimental
 */
export type ListSessionsRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Opaque cursor token from a previous response's nextCursor field for cursor-based pagination
     */
    cursor?: string | null;
    /**
     * Filter sessions by working directory. Must be an absolute path.
     */
    cwd?: string | null;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Response from listing sessions.
 *
 * @experimental
 */
export type ListSessionsResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Opaque cursor token. If present, pass this in the next request's cursor parameter
     * to fetch the next page. If absent, there are no more results.
     */
    nextCursor?: string | null;
    /**
     * Array of session information objects
     */
    sessions: Array<SessionInfo>;
};
/**
 * Request parameters for loading an existing session.
 *
 * Only available if the Agent supports the `loadSession` capability.
 *
 * See protocol docs: [Loading Sessions](https://agentclientprotocol.com/protocol/session-setup#loading-sessions)
 */
export type LoadSessionRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The working directory for this session.
     */
    cwd: string;
    /**
     * List of MCP servers to connect to for this session.
     */
    mcpServers: Array<McpServer>;
    /**
     * The ID of the session to load.
     */
    sessionId: SessionId;
};
/**
 * Response from loading an existing session.
 */
export type LoadSessionResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Initial session configuration options if supported by the Agent.
     *
     * @experimental
     */
    configOptions?: Array<SessionConfigOption> | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Initial model state if supported by the Agent
     *
     * @experimental
     */
    models?: SessionModelState | null;
    /**
     * Initial mode state if supported by the Agent
     *
     * See protocol docs: [Session Modes](https://agentclientprotocol.com/protocol/session-modes)
     */
    modes?: SessionModeState | null;
};
/**
 * MCP capabilities supported by the agent
 */
export type McpCapabilities = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Agent supports [`McpServer::Http`].
     */
    http?: boolean;
    /**
     * Agent supports [`McpServer::Sse`].
     */
    sse?: boolean;
};
/**
 * Configuration for connecting to an MCP (Model Context Protocol) server.
 *
 * MCP servers provide tools and context that the agent can use when
 * processing prompts.
 *
 * See protocol docs: [MCP Servers](https://agentclientprotocol.com/protocol/session-setup#mcp-servers)
 */
export type McpServer = (McpServerHttp & {
    type: "http";
}) | (McpServerSse & {
    type: "sse";
}) | McpServerStdio;
/**
 * HTTP transport configuration for MCP.
 */
export type McpServerHttp = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * HTTP headers to set when making requests to the MCP server.
     */
    headers: Array<HttpHeader>;
    /**
     * Human-readable name identifying this MCP server.
     */
    name: string;
    /**
     * URL to the MCP server.
     */
    url: string;
};
/**
 * SSE transport configuration for MCP.
 */
export type McpServerSse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * HTTP headers to set when making requests to the MCP server.
     */
    headers: Array<HttpHeader>;
    /**
     * Human-readable name identifying this MCP server.
     */
    name: string;
    /**
     * URL to the MCP server.
     */
    url: string;
};
/**
 * Stdio transport configuration for MCP.
 */
export type McpServerStdio = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Command-line arguments to pass to the MCP server.
     */
    args: Array<string>;
    /**
     * Path to the MCP server executable.
     */
    command: string;
    /**
     * Environment variables to set when launching the MCP server.
     */
    env: Array<EnvVariable>;
    /**
     * Human-readable name identifying this MCP server.
     */
    name: string;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * A unique identifier for a model.
 *
 * @experimental
 */
export type ModelId = string;
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Information about a selectable model.
 *
 * @experimental
 */
export type ModelInfo = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Optional description of the model.
     */
    description?: string | null;
    /**
     * Unique identifier for the model.
     */
    modelId: ModelId;
    /**
     * Human-readable name of the model.
     */
    name: string;
};
/**
 * Request parameters for creating a new session.
 *
 * See protocol docs: [Creating a Session](https://agentclientprotocol.com/protocol/session-setup#creating-a-session)
 */
export type NewSessionRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The working directory for this session. Must be an absolute path.
     */
    cwd: string;
    /**
     * List of MCP (Model Context Protocol) servers the agent should connect to.
     */
    mcpServers: Array<McpServer>;
};
/**
 * Response from creating a new session.
 *
 * See protocol docs: [Creating a Session](https://agentclientprotocol.com/protocol/session-setup#creating-a-session)
 */
export type NewSessionResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Initial session configuration options if supported by the Agent.
     *
     * @experimental
     */
    configOptions?: Array<SessionConfigOption> | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Initial model state if supported by the Agent
     *
     * @experimental
     */
    models?: SessionModelState | null;
    /**
     * Initial mode state if supported by the Agent
     *
     * See protocol docs: [Session Modes](https://agentclientprotocol.com/protocol/session-modes)
     */
    modes?: SessionModeState | null;
    /**
     * Unique identifier for the created session.
     *
     * Used in all subsequent requests for this conversation.
     */
    sessionId: SessionId;
};
/**
 * An option presented to the user when requesting permission.
 */
export type PermissionOption = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Hint about the nature of this permission option.
     */
    kind: PermissionOptionKind;
    /**
     * Human-readable label to display to the user.
     */
    name: string;
    /**
     * Unique identifier for this permission option.
     */
    optionId: PermissionOptionId;
};
/**
 * Unique identifier for a permission option.
 */
export type PermissionOptionId = string;
/**
 * The type of permission option being presented to the user.
 *
 * Helps clients choose appropriate icons and UI treatment.
 */
export type PermissionOptionKind = "allow_once" | "allow_always" | "reject_once" | "reject_always";
/**
 * An execution plan for accomplishing complex tasks.
 *
 * Plans consist of multiple entries representing individual tasks or goals.
 * Agents report plans to clients to provide visibility into their execution strategy.
 * Plans can evolve during execution as the agent discovers new requirements or completes tasks.
 *
 * See protocol docs: [Agent Plan](https://agentclientprotocol.com/protocol/agent-plan)
 */
export type Plan = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The list of tasks to be accomplished.
     *
     * When updating a plan, the agent must send a complete list of all entries
     * with their current status. The client replaces the entire plan with each update.
     */
    entries: Array<PlanEntry>;
};
/**
 * A single entry in the execution plan.
 *
 * Represents a task or goal that the assistant intends to accomplish
 * as part of fulfilling the user's request.
 * See protocol docs: [Plan Entries](https://agentclientprotocol.com/protocol/agent-plan#plan-entries)
 */
export type PlanEntry = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Human-readable description of what this task aims to accomplish.
     */
    content: string;
    /**
     * The relative importance of this task.
     * Used to indicate which tasks are most critical to the overall goal.
     */
    priority: PlanEntryPriority;
    /**
     * Current execution status of this task.
     */
    status: PlanEntryStatus;
};
/**
 * Priority levels for plan entries.
 *
 * Used to indicate the relative importance or urgency of different
 * tasks in the execution plan.
 * See protocol docs: [Plan Entries](https://agentclientprotocol.com/protocol/agent-plan#plan-entries)
 */
export type PlanEntryPriority = "high" | "medium" | "low";
/**
 * Status of a plan entry in the execution flow.
 *
 * Tracks the lifecycle of each task from planning through completion.
 * See protocol docs: [Plan Entries](https://agentclientprotocol.com/protocol/agent-plan#plan-entries)
 */
export type PlanEntryStatus = "pending" | "in_progress" | "completed";
/**
 * Prompt capabilities supported by the agent in `session/prompt` requests.
 *
 * Baseline agent functionality requires support for [`ContentBlock::Text`]
 * and [`ContentBlock::ResourceLink`] in prompt requests.
 *
 * Other variants must be explicitly opted in to.
 * Capabilities for different types of content in prompt requests.
 *
 * Indicates which content types beyond the baseline (text and resource links)
 * the agent can process.
 *
 * See protocol docs: [Prompt Capabilities](https://agentclientprotocol.com/protocol/initialization#prompt-capabilities)
 */
export type PromptCapabilities = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Agent supports [`ContentBlock::Audio`].
     */
    audio?: boolean;
    /**
     * Agent supports embedded context in `session/prompt` requests.
     *
     * When enabled, the Client is allowed to include [`ContentBlock::Resource`]
     * in prompt requests for pieces of context that are referenced in the message.
     */
    embeddedContext?: boolean;
    /**
     * Agent supports [`ContentBlock::Image`].
     */
    image?: boolean;
};
/**
 * Request parameters for sending a user prompt to the agent.
 *
 * Contains the user's message and any additional context.
 *
 * See protocol docs: [User Message](https://agentclientprotocol.com/protocol/prompt-turn#1-user-message)
 */
export type PromptRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The blocks of content that compose the user's message.
     *
     * As a baseline, the Agent MUST support [`ContentBlock::Text`] and [`ContentBlock::ResourceLink`],
     * while other variants are optionally enabled via [`PromptCapabilities`].
     *
     * The Client MUST adapt its interface according to [`PromptCapabilities`].
     *
     * The client MAY include referenced pieces of context as either
     * [`ContentBlock::Resource`] or [`ContentBlock::ResourceLink`].
     *
     * When available, [`ContentBlock::Resource`] is preferred
     * as it avoids extra round-trips and allows the message to include
     * pieces of context from sources the agent may not have access to.
     */
    prompt: Array<ContentBlock>;
    /**
     * The ID of the session to send this user message to
     */
    sessionId: SessionId;
};
/**
 * Response from processing a user prompt.
 *
 * See protocol docs: [Check for Completion](https://agentclientprotocol.com/protocol/prompt-turn#4-check-for-completion)
 */
export type PromptResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Indicates why the agent stopped processing the turn.
     */
    stopReason: StopReason;
};
/**
 * Protocol version identifier.
 *
 * This version is only bumped for breaking changes.
 * Non-breaking changes should be introduced via capabilities.
 */
export type ProtocolVersion = number;
/**
 * Request to read content from a text file.
 *
 * Only available if the client supports the `fs.readTextFile` capability.
 */
export type ReadTextFileRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Maximum number of lines to read.
     */
    limit?: number | null;
    /**
     * Line number to start reading from (1-based).
     */
    line?: number | null;
    /**
     * Absolute path to the file to read.
     */
    path: string;
    /**
     * The session ID for this request.
     */
    sessionId: SessionId;
};
/**
 * Response containing the contents of a text file.
 */
export type ReadTextFileResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    content: string;
};
/**
 * Request to release a terminal and free its resources.
 */
export type ReleaseTerminalRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The session ID for this request.
     */
    sessionId: SessionId;
    /**
     * The ID of the terminal to release.
     */
    terminalId: string;
};
/**
 * Response to terminal/release method
 */
export type ReleaseTerminalResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
};
/**
 * JSON RPC Request Id
 *
 * An identifier established by the Client that MUST contain a String, Number, or NULL value if included. If it is not included it is assumed to be a notification. The value SHOULD normally not be Null [1] and Numbers SHOULD NOT contain fractional parts [2]
 *
 * The Server MUST reply with the same value in the Response object if included. This member is used to correlate the context between the two objects.
 *
 * [1] The use of Null as a value for the id member in a Request object is discouraged, because this specification uses a value of Null for Responses with an unknown id. Also, because JSON-RPC 1.0 uses an id value of Null for Notifications this could cause confusion in handling.
 *
 * [2] Fractional parts may be problematic, since many decimal fractions cannot be represented exactly as binary fractions.
 */
export type RequestId = null | bigint | string;
/**
 * The outcome of a permission request.
 */
export type RequestPermissionOutcome = {
    outcome: "cancelled";
} | (SelectedPermissionOutcome & {
    outcome: "selected";
});
/**
 * Request for user permission to execute a tool call.
 *
 * Sent when the agent needs authorization before performing a sensitive operation.
 *
 * See protocol docs: [Requesting Permission](https://agentclientprotocol.com/protocol/tool-calls#requesting-permission)
 */
export type RequestPermissionRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Available permission options for the user to choose from.
     */
    options: Array<PermissionOption>;
    /**
     * The session ID for this request.
     */
    sessionId: SessionId;
    /**
     * Details about the tool call requiring permission.
     */
    toolCall: ToolCallUpdate;
};
/**
 * Response to a permission request.
 */
export type RequestPermissionResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The user's decision on the permission request.
     */
    outcome: RequestPermissionOutcome;
};
/**
 * A resource that the server is capable of reading, included in a prompt or tool call result.
 */
export type ResourceLink = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    annotations?: Annotations | null;
    description?: string | null;
    mimeType?: string | null;
    name: string;
    size?: bigint | null;
    title?: string | null;
    uri: string;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Request parameters for resuming an existing session.
 *
 * Resumes an existing session without returning previous messages (unlike `session/load`).
 * This is useful for agents that can resume sessions but don't implement full session loading.
 *
 * Only available if the Agent supports the `session.resume` capability.
 *
 * @experimental
 */
export type ResumeSessionRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The working directory for this session.
     */
    cwd: string;
    /**
     * List of MCP servers to connect to for this session.
     */
    mcpServers?: Array<McpServer>;
    /**
     * The ID of the session to resume.
     */
    sessionId: SessionId;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Response from resuming an existing session.
 *
 * @experimental
 */
export type ResumeSessionResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Initial session configuration options if supported by the Agent.
     *
     * @experimental
     */
    configOptions?: Array<SessionConfigOption> | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Initial model state if supported by the Agent
     *
     * @experimental
     */
    models?: SessionModelState | null;
    /**
     * Initial mode state if supported by the Agent
     *
     * See protocol docs: [Session Modes](https://agentclientprotocol.com/protocol/session-modes)
     */
    modes?: SessionModeState | null;
};
/**
 * The sender or recipient of messages and data in a conversation.
 */
export type Role = "assistant" | "user";
/**
 * The user selected one of the provided options.
 */
export type SelectedPermissionOutcome = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The ID of the option the user selected.
     */
    optionId: PermissionOptionId;
};
/**
 * Session capabilities supported by the agent.
 *
 * As a baseline, all Agents **MUST** support `session/new`, `session/prompt`, `session/cancel`, and `session/update`.
 *
 * Optionally, they **MAY** support other session methods and notifications by specifying additional capabilities.
 *
 * Note: `session/load` is still handled by the top-level `load_session` capability. This will be unified in future versions of the protocol.
 *
 * See protocol docs: [Session Capabilities](https://agentclientprotocol.com/protocol/initialization#session-capabilities)
 */
export type SessionCapabilities = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Whether the agent supports `session/fork`.
     *
     * @experimental
     */
    fork?: SessionForkCapabilities | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Whether the agent supports `session/list`.
     *
     * @experimental
     */
    list?: SessionListCapabilities | null;
    /**
     * **UNSTABLE**
     *
     * This capability is not part of the spec yet, and may be removed or changed at any point.
     *
     * Whether the agent supports `session/resume`.
     *
     * @experimental
     */
    resume?: SessionResumeCapabilities | null;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Unique identifier for a session configuration option value group.
 *
 * @experimental
 */
export type SessionConfigGroupId = string;
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Unique identifier for a session configuration option.
 *
 * @experimental
 */
export type SessionConfigId = string;
export type SessionConfigOption = SessionConfigSelect & {
    type: "select";
} & {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Optional description for the Client to display to the user.
     */
    description?: string | null;
    /**
     * Unique identifier for the configuration option.
     */
    id: SessionConfigId;
    /**
     * Human-readable label for the option.
     */
    name: string;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * A single-value selector (dropdown) session configuration option payload.
 *
 * @experimental
 */
export type SessionConfigSelect = {
    /**
     * The currently selected value.
     */
    currentValue: SessionConfigValueId;
    /**
     * The set of selectable options.
     */
    options: SessionConfigSelectOptions;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * A group of possible values for a session configuration option.
 *
 * @experimental
 */
export type SessionConfigSelectGroup = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Unique identifier for this group.
     */
    group: SessionConfigGroupId;
    /**
     * Human-readable label for this group.
     */
    name: string;
    /**
     * The set of option values in this group.
     */
    options: Array<SessionConfigSelectOption>;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * A possible value for a session configuration option.
 *
 * @experimental
 */
export type SessionConfigSelectOption = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Optional description for this option value.
     */
    description?: string | null;
    /**
     * Human-readable label for this option value.
     */
    name: string;
    /**
     * Unique identifier for this option value.
     */
    value: SessionConfigValueId;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Possible values for a session configuration option.
 *
 * @experimental
 */
export type SessionConfigSelectOptions = Array<SessionConfigSelectOption> | Array<SessionConfigSelectGroup>;
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Unique identifier for a session configuration option value.
 *
 * @experimental
 */
export type SessionConfigValueId = string;
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Capabilities for the `session/fork` method.
 *
 * By supplying `{}` it means that the agent supports forking of sessions.
 *
 * @experimental
 */
export type SessionForkCapabilities = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
};
/**
 * A unique identifier for a conversation session between a client and agent.
 *
 * Sessions maintain their own context, conversation history, and state,
 * allowing multiple independent interactions with the same agent.
 *
 * See protocol docs: [Session ID](https://agentclientprotocol.com/protocol/session-setup#session-id)
 */
export type SessionId = string;
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Information about a session returned by session/list
 *
 * @experimental
 */
export type SessionInfo = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The working directory for this session. Must be an absolute path.
     */
    cwd: string;
    /**
     * Unique identifier for the session
     */
    sessionId: SessionId;
    /**
     * Human-readable title for the session
     */
    title?: string | null;
    /**
     * ISO 8601 timestamp of last activity
     */
    updatedAt?: string | null;
};
/**
 * Update to session metadata. All fields are optional to support partial updates.
 *
 * Agents send this notification to update session information like title or custom metadata.
 * This allows clients to display dynamic session names and track session state changes.
 */
export type SessionInfoUpdate = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Human-readable title for the session. Set to null to clear.
     */
    title?: string | null;
    /**
     * ISO 8601 timestamp of last activity. Set to null to clear.
     */
    updatedAt?: string | null;
};
/**
 * Capabilities for the `session/list` method.
 *
 * By supplying `{}` it means that the agent supports listing of sessions.
 *
 * Further capabilities can be added in the future for other means of filtering or searching the list.
 */
export type SessionListCapabilities = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
};
/**
 * A mode the agent can operate in.
 *
 * See protocol docs: [Session Modes](https://agentclientprotocol.com/protocol/session-modes)
 */
export type SessionMode = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    description?: string | null;
    id: SessionModeId;
    name: string;
};
/**
 * Unique identifier for a Session Mode.
 */
export type SessionModeId = string;
/**
 * The set of modes and the one currently active.
 */
export type SessionModeState = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The set of modes that the Agent can operate in
     */
    availableModes: Array<SessionMode>;
    /**
     * The current mode the Agent is in.
     */
    currentModeId: SessionModeId;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * The set of models and the one currently active.
 *
 * @experimental
 */
export type SessionModelState = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The set of models that the Agent can use
     */
    availableModels: Array<ModelInfo>;
    /**
     * The current model the Agent is in.
     */
    currentModelId: ModelId;
};
/**
 * Notification containing a session update from the agent.
 *
 * Used to stream real-time progress and results during prompt processing.
 *
 * See protocol docs: [Agent Reports Output](https://agentclientprotocol.com/protocol/prompt-turn#3-agent-reports-output)
 */
export type SessionNotification = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The ID of the session this update pertains to.
     */
    sessionId: SessionId;
    /**
     * The actual update content.
     */
    update: SessionUpdate;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Capabilities for the `session/resume` method.
 *
 * By supplying `{}` it means that the agent supports resuming of sessions.
 *
 * @experimental
 */
export type SessionResumeCapabilities = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
};
/**
 * Different types of updates that can be sent during session processing.
 *
 * These updates provide real-time feedback about the agent's progress.
 *
 * See protocol docs: [Agent Reports Output](https://agentclientprotocol.com/protocol/prompt-turn#3-agent-reports-output)
 */
export type SessionUpdate = (ContentChunk & {
    sessionUpdate: "user_message_chunk";
}) | (ContentChunk & {
    sessionUpdate: "agent_message_chunk";
}) | (ContentChunk & {
    sessionUpdate: "agent_thought_chunk";
}) | (ToolCall & {
    sessionUpdate: "tool_call";
}) | (ToolCallUpdate & {
    sessionUpdate: "tool_call_update";
}) | (Plan & {
    sessionUpdate: "plan";
}) | (AvailableCommandsUpdate & {
    sessionUpdate: "available_commands_update";
}) | (CurrentModeUpdate & {
    sessionUpdate: "current_mode_update";
}) | (ConfigOptionUpdate & {
    sessionUpdate: "config_option_update";
}) | (SessionInfoUpdate & {
    sessionUpdate: "session_info_update";
});
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Request parameters for setting a session configuration option.
 *
 * @experimental
 */
export type SetSessionConfigOptionRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The ID of the configuration option to set.
     */
    configId: SessionConfigId;
    /**
     * The ID of the session to set the configuration option for.
     */
    sessionId: SessionId;
    /**
     * The ID of the configuration option value to set.
     */
    value: SessionConfigValueId;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Response to `session/set_config_option` method.
 *
 * @experimental
 */
export type SetSessionConfigOptionResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The full set of configuration options and their current values.
     */
    configOptions: Array<SessionConfigOption>;
};
/**
 * Request parameters for setting a session mode.
 */
export type SetSessionModeRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The ID of the mode to set.
     */
    modeId: SessionModeId;
    /**
     * The ID of the session to set the mode for.
     */
    sessionId: SessionId;
};
/**
 * Response to `session/set_mode` method.
 */
export type SetSessionModeResponse = {
    _meta?: {
        [key: string]: unknown;
    } | null;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Request parameters for setting a session model.
 *
 * @experimental
 */
export type SetSessionModelRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The ID of the model to set.
     */
    modelId: ModelId;
    /**
     * The ID of the session to set the model for.
     */
    sessionId: SessionId;
};
/**
 * **UNSTABLE**
 *
 * This capability is not part of the spec yet, and may be removed or changed at any point.
 *
 * Response to `session/set_model` method.
 *
 * @experimental
 */
export type SetSessionModelResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
};
/**
 * Reasons why an agent stops processing a prompt turn.
 *
 * See protocol docs: [Stop Reasons](https://agentclientprotocol.com/protocol/prompt-turn#stop-reasons)
 */
export type StopReason = "end_turn" | "max_tokens" | "max_turn_requests" | "refusal" | "cancelled";
/**
 * Embed a terminal created with `terminal/create` by its id.
 *
 * The terminal must be added before calling `terminal/release`.
 *
 * See protocol docs: [Terminal](https://agentclientprotocol.com/protocol/terminals)
 */
export type Terminal = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    terminalId: string;
};
/**
 * Exit status of a terminal command.
 */
export type TerminalExitStatus = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The process exit code (may be null if terminated by signal).
     */
    exitCode?: number | null;
    /**
     * The signal that terminated the process (may be null if exited normally).
     */
    signal?: string | null;
};
/**
 * Request to get the current output and status of a terminal.
 */
export type TerminalOutputRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The session ID for this request.
     */
    sessionId: SessionId;
    /**
     * The ID of the terminal to get output from.
     */
    terminalId: string;
};
/**
 * Response containing the terminal output and exit status.
 */
export type TerminalOutputResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Exit status if the command has completed.
     */
    exitStatus?: TerminalExitStatus | null;
    /**
     * The terminal output captured so far.
     */
    output: string;
    /**
     * Whether the output was truncated due to byte limits.
     */
    truncated: boolean;
};
/**
 * Text provided to or from an LLM.
 */
export type TextContent = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    annotations?: Annotations | null;
    text: string;
};
/**
 * Text-based resource contents.
 */
export type TextResourceContents = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    mimeType?: string | null;
    text: string;
    uri: string;
};
/**
 * Represents a tool call that the language model has requested.
 *
 * Tool calls are actions that the agent executes on behalf of the language model,
 * such as reading files, executing code, or fetching data from external sources.
 *
 * See protocol docs: [Tool Calls](https://agentclientprotocol.com/protocol/tool-calls)
 */
export type ToolCall = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Content produced by the tool call.
     */
    content?: Array<ToolCallContent>;
    /**
     * The category of tool being invoked.
     * Helps clients choose appropriate icons and UI treatment.
     */
    kind?: ToolKind;
    /**
     * File locations affected by this tool call.
     * Enables "follow-along" features in clients.
     */
    locations?: Array<ToolCallLocation>;
    /**
     * Raw input parameters sent to the tool.
     */
    rawInput?: unknown;
    /**
     * Raw output returned by the tool.
     */
    rawOutput?: unknown;
    /**
     * Current execution status of the tool call.
     */
    status?: ToolCallStatus;
    /**
     * Human-readable title describing what the tool is doing.
     */
    title: string;
    /**
     * Unique identifier for this tool call within the session.
     */
    toolCallId: ToolCallId;
};
/**
 * Content produced by a tool call.
 *
 * Tool calls can produce different types of content including
 * standard content blocks (text, images) or file diffs.
 *
 * See protocol docs: [Content](https://agentclientprotocol.com/protocol/tool-calls#content)
 */
export type ToolCallContent = (Content & {
    type: "content";
}) | (Diff & {
    type: "diff";
}) | (Terminal & {
    type: "terminal";
});
/**
 * Unique identifier for a tool call within a session.
 */
export type ToolCallId = string;
/**
 * A file location being accessed or modified by a tool.
 *
 * Enables clients to implement "follow-along" features that track
 * which files the agent is working with in real-time.
 *
 * See protocol docs: [Following the Agent](https://agentclientprotocol.com/protocol/tool-calls#following-the-agent)
 */
export type ToolCallLocation = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Optional line number within the file.
     */
    line?: number | null;
    /**
     * The file path being accessed or modified.
     */
    path: string;
};
/**
 * Execution status of a tool call.
 *
 * Tool calls progress through different statuses during their lifecycle.
 *
 * See protocol docs: [Status](https://agentclientprotocol.com/protocol/tool-calls#status)
 */
export type ToolCallStatus = "pending" | "in_progress" | "completed" | "failed";
/**
 * An update to an existing tool call.
 *
 * Used to report progress and results as tools execute. All fields except
 * the tool call ID are optional - only changed fields need to be included.
 *
 * See protocol docs: [Updating](https://agentclientprotocol.com/protocol/tool-calls#updating)
 */
export type ToolCallUpdate = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * Replace the content collection.
     */
    content?: Array<ToolCallContent> | null;
    /**
     * Update the tool kind.
     */
    kind?: ToolKind | null;
    /**
     * Replace the locations collection.
     */
    locations?: Array<ToolCallLocation> | null;
    /**
     * Update the raw input.
     */
    rawInput?: unknown;
    /**
     * Update the raw output.
     */
    rawOutput?: unknown;
    /**
     * Update the execution status.
     */
    status?: ToolCallStatus | null;
    /**
     * Update the human-readable title.
     */
    title?: string | null;
    /**
     * The ID of the tool call being updated.
     */
    toolCallId: ToolCallId;
};
/**
 * Categories of tools that can be invoked.
 *
 * Tool kinds help clients choose appropriate icons and optimize how they
 * display tool execution progress.
 *
 * See protocol docs: [Creating](https://agentclientprotocol.com/protocol/tool-calls#creating)
 */
export type ToolKind = "read" | "edit" | "delete" | "move" | "search" | "execute" | "think" | "fetch" | "switch_mode" | "other";
/**
 * All text that was typed after the command name is provided as input.
 */
export type UnstructuredCommandInput = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * A hint to display when the input hasn't been provided yet
     */
    hint: string;
};
/**
 * Request to wait for a terminal command to exit.
 */
export type WaitForTerminalExitRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The session ID for this request.
     */
    sessionId: SessionId;
    /**
     * The ID of the terminal to wait for.
     */
    terminalId: string;
};
/**
 * Response containing the exit status of a terminal command.
 */
export type WaitForTerminalExitResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The process exit code (may be null if terminated by signal).
     */
    exitCode?: number | null;
    /**
     * The signal that terminated the process (may be null if exited normally).
     */
    signal?: string | null;
};
/**
 * Request to write content to a text file.
 *
 * Only available if the client supports the `fs.writeTextFile` capability.
 */
export type WriteTextFileRequest = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
    /**
     * The text content to write to the file.
     */
    content: string;
    /**
     * Absolute path to the file to write.
     */
    path: string;
    /**
     * The session ID for this request.
     */
    sessionId: SessionId;
};
/**
 * Response to `fs/write_text_file`
 */
export type WriteTextFileResponse = {
    /**
     * The _meta property is reserved by ACP to allow clients and agents to attach additional
     * metadata to their interactions. Implementations MUST NOT make assumptions about values at
     * these keys.
     *
     * See protocol docs: [Extensibility](https://agentclientprotocol.com/protocol/extensibility)
     */
    _meta?: {
        [key: string]: unknown;
    } | null;
};
