#*******************************************************************************
# E.S.O. - VLT project
#
# "@(#) $Id: bash_library.sh 289259 2016-10-16 02:24:35Z pbaksai $"
#
# who       when        what
# --------  ----------  --------------------------------------------------------
# pbaksai   2016/10/15  Fixed problem with color attribute escape codes.
# pbaksai   2016/07/07  Improved attribute management for formatted output.
# pbaksai   2015/11/12  CCB-000221: Improved ParVal.
# pbaksai   2015/06/14  CCB-000221: Added TclTkVersion, CppVersion.
#                                   Improved ShellVersion.
# pbaksai   2015/06/14  CCB-000755: Fixed CheckRepository for new format in
#                                   svn 1.8.
# pbaksai   2014/09/20  CCB-000755: Fixed /tmp/__cmd_error__ access.
# pbaksai   2014/07/27  CCB-000221: Improved automatic error reporting.
# pbaksai   2014/07/10  CCB-000579: Fixed portable tar command.
# pbaksai   2014/07/01  CCB-000579: Added portable tar command.
#                                   Fixed tests for integer and real ranges.
# pbaksai   2014/06/30  CCB-000579: Corrected log file path.
# pbaksai   2014/06/27  CCB-000579: Fixed typo and documentation.
# pbaksai   2014/06/25  CCB-000579: Improved architecture detection.
# pbaksai   2014/06/13  CCB-000221: Improved logging.
# pbaksai   2014/04/21  CCB-000579: Fixed dependency problem in Period function.
#                                   Mitigated effect of bug in resize command.
#                                   Added portable stty_size option.
# pbaksai   2014/01/28  CCB-000579: Added CheckRange function.
# pbaksai   2014/02/23  CCB-000579: Minor corrections and improvements.
# pbaksai   2014/01/28  CCB-000579: Added portable ls_nobackup option.
# pbaksai   2014/01/25  CCB-000221: Improved terminal columns, lines.
#                                   Fixed bug in Version().
# pbaksai   2013/12/30  CCB-000221: Fixed typos in date and time constant name.
# pbaksai   2013/12/28  CCB-000221: Corrections for portability.
# pbaksai   2013/10/17  CCB-000221: Created.
#
#*******************************************************************************
#   NAME
#       bash_library.sh - Library for BASH Scripts.
#
#   SYNOPSIS
#       source bash_library.sh
#
#   DESCRIPTION
#       Library of BASH functions, global variables, and constants.
#
#   FUNCTIONS
#       Alive:           Checks if the machine is responding.
#       Architecture:    Determines the kernel architecture.
#       SystemType:      Determines the operating system type.
#       TargetType:      Determines remote target operating system type.
#       GetELF:          Reads the Executable Linkable Format of an object.
#       CppVersion:      Checks the C and C++ compiler version.
#       ShellVersion:    Checks the BASH shell version.
#       TclTkVersion:    Checks the Tcl/Tk version.
#       Portable:        Resolves portability issues according to current system
#       SignalHandlers:  Installs signal handlers.
#       Sleep:           Suspends the execution for a number of seconds.
#       Spinner:         Prints a spinning character at the current cursor.
#       Waitfor:         Prints a message and a spinner while a command executes
#       Period:          Calculates which observation period a date belongs to.
#       LastDay:         Returns the number of the last day of each month.
#       Yesterday:       Returns yesterdays date in YYYY-MM-DD format.
#       StartLog:        Prepares log file to receive output from the program.
#       HelpAd:          Prints a message explaining how to get help.
#       Help:            Prints detailed instructions on how to run the program.
#       Items:           Prints the list of accepted values of a parameter.
#       Version:         Prints the program's revision, date, and author.
#       GetOpt:          Emulates pure POSIX getopt for all OS flavors.
#           _getopt_main
#           _getopt_parse
#           _getopt_err
#           _getopt_resolve_abbrev
#           _getopt_split
#           _getopt_quote
#           _getopt_quote_csh
#           _getopt_help
#       ExpandRange:     Transforms a range into a sequence.
#       CheckRange:      Checks if a value is within a specific range or not.
#       ParNum:          Prints an error message about the number of parameters.
#       ParVal:          Prints detailed message about invalid parameter values.
#       CheckRepository: Checks that inside a working copy and access to SVN.
#       Delete:          Deletes an external variable's file.
#       Export:          Saves the external variable's value in a file.
#       Import:          Returns the external variable's file contents.
#       Stack:           Returns the current size of the stack for next push.
#       Pop:             Pops the top stack slot.
#       Push:            Pushes data to the top of the stack.
#       CleanUp:         Performs all cleaning actions upon termination.
#       StopProcs:       Suspends registered background processes.
#       KillProcs:       Kills registered background processes.
#       CheckStatus:     Checks the status code from $? and prints a message.
#       Exit:            Standard exit for program. Signal handler.
#       GetAnswer:       Receives user input with timeout, default, help, menu.
#       PrintLine:       Prints a message in one or more justified lines.
#       Debug:           Prints a warning message in one or more justified lines
#       Error:           Prints an error message in one or more justified lines.
#       Warning:         Prints a warning message in one or more justified lines
#       Explain:         Prints a continuation message in one or more lines.
#       Doing:           Prints an opening message for a task section.
#       Done:            Prints a closing message for a task section.
#       Tell:            Prints an intermediate message in a task section.
#       Banner:          Prints a message in one or more justified lines.
#       Filler:          Prints an empty justified line within margins.
#       Frame:           Prints a line of replicated characters within margins.
#
#   DATA MEMBERS
#       Environment
#         user:          User name.
#         ws:            Host name.
#         hd:            Home directory.
#         bv:            Bash version.
#         log:           Flag from environment variable enabling log to file.
#         dbg:           Flag from environment variable enabling debug messages.
#       Global
#         progname:      Program name.
#         progpath:      Program path.
#         datestamp:     Date string YYYY-MM-DD.
#         yearstamp:     Year string YYYY.
#         timestamp:     Time stamp HH:MM:SS.
#         save_tty:      Current terminal configuration.
#         dflag:         Flag from options enabling debugging messages.
#         eflag:         Error flag or code.
#         lflag:         Flag from options enabling log to file.
#         lfile:         Log filename.
#         endian:        Endianess test result: 0 (big endian) or 1 (little endian).
#         FGCOLOR:       Color used for next text display.
#         PID:           Array storing process id used.
#         p:             Index to next slot available in PID.
#         VAR:           Array of external variables used.
#         v:             Index to next slot available in VAR.
#         LIT:           Array of ordered literals.
#         l:             Index to LIT.
#       Portable command placeholders:
#         bash:
#         df:
#         ping:
#         remote:
#         tar:
#         xterm:
#       Portable command option placeholders:
#         date_date:
#         echo_c:
#         echo_n:
#         echo_escape:
#         echo_newline:
#         ls_ignore:
#         ls_nobackup:
#         ping_wait:
#         ping_count:
#         ping_hpcount:
#         count:
#         read_timeout:
#         stty_size:
#         tar_exclude:
#         uname_machine:
#       Portable flags:
#         intsec: if true, only integer seconds as argument to sleep.
#         intusec: if true, integer micro seconds as argument to usleep.
#         epochf: UNIX epoch format, is @ in Linux, %s in Darwin.
#         dateif: Input date format, is required for date -j -f $dateif $date
#                 in Darwin.
#         e: escape code for using in echo -e ${e}[ color and attribute codes.
#       Constants
#         Hardcoded
#          SHELL_OLDEST_VERSION="2.03"
#          DF='%Y-%m-%d' Date format.
#          YF='%Y'       Year format.
#          TF='%H:%M:%S' Time format.
#          LIFS=';'      Local Internal Field Separator.
#          LSEP='='      Local Separator.
#          MINLIN=70     Minimum number of lines in the terminal.
#          MINCOL=80     Minimum number of columns in the terminal.
#          LMARGIN=5     Number of columns on the left margin.
#          RMARGIN=5     Number of columns in the right margin.
#          FG_WARN       Foreground code for warning messages.
#          FG_ERROR      Foreground code for error messages.
#          FG_DEBUG      Foreground code for debugging messages.
#          EC_END=0      Exit code for normal end.
#          EC_CANCEL=1   Exit code  cancel.
#          EC_ABORT=2    Exit code for abort.
#          EC_REJECT=3   Exit code for invalid arguments.
#          EC_INFO=4     Exit code after help, usage, version, license display.
#          EC_QUIET=9    Exit code for end skipping cleanup and final banner.
#          MININT32      Lower limit of integer arithmetics.
#          MAXINT32      Upper limit of integer arithmetics.
#         Derived
#          UNAME         System type.
#          OSREL         Operating system release.
#          OSTAG         Operating system tag.
#          CPUHW         Kernel architecture.
#          BITS          Bit width of the kernel architecture.
#          BIGENDIAN     Endianess flag.
#          LINCOL        Current terminal size in lines and columns.
#          LINES         Number of columns in the current terminal size.
#          COLUMNS       Number of columns in the current terminal size.
#          MAXLIN        Maximum number of lines in the terminal size.
#          MAXCOL        Maximum number of columns in the terminal size.
#          MARGIN        Left margin blank prefix.
#          SEPARATOR     A line made of repeated character as text separator.
#          LOGFILE       Default log file (<progname>.log).
#          FGC_LIST      List of the following 9 foreground colors.
#          BLACK         Foreground colors.
#          RED
#          GREEN
#          YELLOW
#          BLUE
#          MAGENTA
#          CYAN
#          GRAY
#          OLIVE
#          TAT_LIST      List of the following 11 text attributes.
#          BOLD          Text attributes.
#          BLINK
#          HIDDEN
#          REVERSE
#          UNDERLINED
#          RESET
#          RESETBOLD
#          RESETBLINK
#          RESETHIDDEN
#          RESETREVERSE
#          RESETUNDERLINE
#          P1            Patterns to match the echo escape codes.
#          P2
#          P3
#          P4
#          HELP_LINE     Default parameter syntax.
#          USAGE_LINE    Default parameter description.
#          LICENSE_LINE  <GPL v3>.
#
#   PARAMETERS
#       None
#
#   ARGUMENTS
#       None
#
#   INPUTS
#       None
#
#   OUTPUTS
#       None
#
#   FILES
#       /tmp/__cmd_error__
#
#   ENVIRONMENT
#       USER
#       HOST
#       HOME
#       SHELL_LOG
#       SHELL_DEBUG
#
#   RETURN VALUES
#       None
#
#   CAUTIONS
#       Programs using this library can redefine (overload) functions as needed
#       and adjust constants after sourcing this code.
#
#   EXAMPLES
#       None
#
#   SEE ALSO
#       None
#
#   BUGS
#       None
#
#-------------------------------------------------------------------------------
#        1         2         3         4         5         6         7         8
#23456789 123456789 123456789 123456789 123456789 123456789 123456789 1234567890

########################
# Standard Library BEGIN
#
#********#
# System #
#********#
#
# Alive: Checks if the machine is responding.
#
Alive() {
    # Parameters:
    # 1: Target name or IP address
    # Globals:
    # - ping: ping command according to local OS flavor
    # - ping_count: option for ping command according to local OS flavor
    # - ping_hpcount: option for HP-UX ping command required after host arg
    # - ping_wait: option for ping command according to local OS flavor
    # - remote: remote shell command according to local OS flavor
    # Assigns:    nothing
    # Returns:
    # - 0: target is alive
    # - 1: target is not alive
    local __ping_wait=${ping_wait//10/1}
    local __name="$1"
    local __alive=0
    __alive=`$ping $__ping_wait $ping_count $__name $ping_hpcount 2>&- | awk '/ 0%|100%| transmitted/{printf("%s", $4)}'`
    if [ -z "$__alive" ]; then
        __alive=0
    fi
    if [ $__alive -ne 1 ]; then
        echo 1
        return 1
    fi
    echo 0
    return 0
}

#
# Architecture: Determines the machine architecture.
#
Architecture() {
    # Parameters:
    # 1: CPU hardware (architecture, 32, 64 bits)
    # Globals:    none
    # Assigns:    nothing
    # Returns:
    # - Processor architecture: includes corresponding _32 or _64 suffix
    local __cpuhw="$1"
    case "$__cpuhw" in
        i[3456789]86 | \
        x86 | \
        i86pc )
            # Intel or AMD 32 bits
            echo x86_32
            ;;
        x86_64 | \
        AMD64 | \
        amd64 )
            # Intel or AMD 64 bits
            echo x86_64
            ;;
        9000* | \
        IA64 | \
        ia64 )
            # HP IA64 (Intel Architecture 64 or Itanium)
            echo risc_`getconf KERNEL_BITS`
            ;;
        sparc )
            # Oracle Sun 32 bits
            echo sparc_32
            ;;
        sparc_64 )
            # Oracle Sun 64 bits
            echo sparc_64
            ;;
        ppc )
            # Motorola 32 bits
            echo ppc_32
            ;;
        ppc64 )
            # Motorola 64 bits
            echo ppc_64
            ;;
        (*)
            # 
            echo "UNKNOWN_$__cpuhw"
            ;;
    esac
}

#
# SystemType: Determines the operating system type.
#
SystemType() {
    # Parameters:
    # 1: Operating system name
    # 2: Operating system release
    # 3: System name
    # Globals:    none
    # Assigns:    nothing
    # Returns:
    # - System tag:
    #   - AIX
    #   - DEC
    #   - HPU
    #   - LIN
    #   - MAC
    #   - OSF
    #   - SCO
    #   - SGI
    #   - SUNBSD
    #   - SOLARIS
    #   - SYSTEMV
    #   - UNKNOWN
    local __uname="$1"
    local __osrel="$2"
    local __hostn=`echo $3 | sed 's/\..*//'`
    case "$__uname" in
        AIX )
            # IBM
            echo AIX
            ;;
        Darwin )
            # Apple MAC
            echo MAC
            ;;
        HP-UX )
            # Hewllet Packard
            echo HPU
            ;;
        IRIX )
            # Silicon Graphics
            echo SGI
            ;;
        Linux )
            # 
            echo LIN
            ;;
        OSF1 )
            # Digital Equipmen (DEC) newer OSF system
            echo OSF
            ;;
        ULTRIX )
            # Digital Equipment (DEC) old BSD system
            echo DEC
            ;;
        SunOS )
            # Sun BSD or System V (Solaris)
            case "$__osrel" in
                4*) 
                    echo SUNBSD
                    ;;
                5*) 
                    echo SOLARIS
                    ;;
                (*) 
                    echo SYSTEMV
                    ;;
            esac
            ;;
        $__hostn )
            # Some old systems return the user defined name of the system
            # instead of the name of the operating system: SGI's IRIX 3.3
            # and before, and SCO's 3.2
            local __cpuhw=`uname -m`
            case "$__cpuhw" in
                IP*)
                    echo SGI
                    ;;
                i386)
                    echo SCO
                    ;;
                (*)
                    echo "UNKNOWN $__uname"
                    ;;
            esac
            ;;
        (*)
            # 
            echo "UNKNOWN $__uname"
            ;;
    esac
}

#
# TargetType: Finds the remote target OS type.
#
TargetType() {
    # Parameters:
    # 1: Target name or IP address
    # Globals:
    # - remote: Command for remote shell access
    # Assigns:    nothing
    # Returns:
    # - type: Remote OS type
    local __type=UNKNOWN
    local __stat=0
    if [ "${1:0:1}" = "l" ]; then
        __type=LCU
    else
        __stat=$(Alive $1)
        if [ $__stat -eq 0 ]; then
            local __uname=`$remote $1 -n "uname -s" 2>&-`
            local __osrel=`$remote $1 -n "uname -r" 2>&-`
            __type=$(SystemType "$__uname" "$__osrel" "$1" )
        fi
    fi
    echo "$__type"
}

#
# GetELF: Reads the Executable Linkable Format of an object
#         to determine if it is 32 or 64 bits.
#
GetELF() {
    # Parameters:
    # 1: Path to the object
    # Globals:    none
    # Assigns:    nothing
    # Returns:
    # - bits: 32 or 64 according to ELF of the object
    #         or empty if not ELF
    local __o="$1"
    local __b=0
    __b=`file -L "$__o" | grep ELF`
    __b=`echo ${__b/*ELF /} | awk -F'-' '{printf("%d",$1)}'`
    echo "$__b"
    return $__b
}

#
# CppVersion: Checks that the C and C++ compiler version is more recent than
#             a minimum required version.
#
CppVersion() {
    # Parameters:
    # 1: Minimum required version
    # Globals:
    # - SHELL_OLDEST_VERSION
    # Assigns:    nothing
    # Returns:
    # - 0: current version is OK
    # - 1: current version is older than minimum required
    # - 2: unknown current version
    local __m="${1##*[!.0-9]*}"
    __m=${__m:-4.1}
    local __p1=$(which gcc 2>/dev/null)
    local __p2=$(which g++ 2>/dev/null)
    if [ -z "$__p1" -a -z "$__p2" ]; then
        Warning "Could not find C/C++ compiler in the path."
        return 2
    else
        local __vCC=`$__p1 -v  2>&1 | grep "gcc version" | awk '{printf("%s",$3)}'`
        local __vPP=`$__p2 -v  2>&1 | grep "gcc version" | awk '{printf("%s",$3)}'`
        if [ -z "$__vCC" -a -z "$__vPP" ]; then
           Warning "Could not find C version."
           return 3
        fi
    fi
    # This is a lexical comparison that should be sufficient forever.
    if [[ "$__vCC" < "$__m" ]]; then
        Error "C $__vCC is not compatible. Cancel." >&2
        Explain "Upgrade to at least $__m" >&2
        return 1
    fi
    return 0
}

#
# ShellVersion: Checks that the BASH shell version is more recent than
#               a minimum required version.
#
ShellVersion() {
    # Parameters:
    # 1: Minimum required version
    # Globals:
    # - SHELL_OLDEST_VERSION
    # Assigns:    nothing
    # Returns:
    # - 0: current version is OK
    # - 1: current version is older than minimum required
    # - 2: unknown current version
    local __m="${1##*[!.0-9]*}"
    __m=${__m:-${SHELL_OLDEST_VERSION}}
    if [ -z "$BASH_VERSION" ]; then
        echo "Unknown version of BASH might not be compatible. Cancel." >&2
        echo "Upgrade to $SHELL_OLDEST_VERSION at least." >&2
        return 2
    fi

    local __v=${BASH_VERSION%%(*}
    # This is a lexical comparison that should be sufficient forever.
    if [[ "$__v" < "$__m" ]]; then
        echo "BASH $__v is not compatible. Cancel." >&2
        echo "Upgrade to at least $__m" >&2
        return 1
    fi
    return 0
}

#
# TclTkVersion: Checks that the TclTk interpreter version is more recent than
#               a minimum required version.
#
TclTkVersion() {
    # Parameters:
    # 1: Minimum required version
    # Globals:
    # - 
    # Assigns:    nothing
    # Returns:
    # - 0: current version is OK
    # - 1: current version is older than minimum required
    # - 2: could not find TCl/Tk ithe path
    # - 3: could not obtain version of Tcl/Tk interpreter
    local __m="${1##*[!.0-9]*}"
    __m=${__m:-8.4}
    local __p=$(which tclsh 2>/dev/null)
    if [ -z "$__p" ]; then
        Warning "Could not find Tcl/Tk interpreter in the path."
        return 2
    else
        local __v=$(echo "puts [info patchlevel]" | $__p)
        if [ -z "$__v" ]; then
           Warning "Could not find Tcl/Tk version."
           return 3
        fi
    fi
    # This is a lexical comparison that should be sufficient forever.
    if [[ "$__v" < "$__m" ]]; then
        Error "Tcl/Tk $__v is not compatible. Cancel." >&2
        Explain "Upgrade to at least $__m" >&2
        return 1
    fi
    return 0
}

#
# Portable: Resolves portability issues according to current system.
#
Portable() {
    # Parameters:
    # 1: Operating system tag
    # 2: Operating system release
    # Globals:    none
    # Assigns:
    # portable commands:
    # - bash:
    # - df:
    # - ping:
    # - remote:
    # - tar:
    # - xterm:
    # portable options:
    # - date_date:
    # - echo_c:
    # - echo_n:
    # - echo_escape:
    # - echo_newline:
    # - ls_ignore:
    # - ls_nobackup:
    # - ping_wait:
    # - ping_count:
    # - ping_hpcount:
    # - count:
    # - read_timeout:
    # - stty_size:
    # - tar_exclude:
    # - uname_machine:
    # portable flags:
    # - intsec: if true, only integer seconds as argument to sleep
    # - intusec: if true, integer micro seconds as argument to usleep
    # - epochf: UNIX epoch format, is @ in Linux, %s in Darwin
    # - dateif: Input date format, is required for date -j -f $dateif $date
    #           in Darwin
    # - e: escape code for using in echo -e ${e}[ color and attribute codes
    # Returns:    nothing
    local __sname="$1"
    local __osrel="$2"

    # Resolve options for 'echo'
    # - no_newline
    if [ "`echo -n`" = "-n" ]; then
        echo_c='\c'
        echo_n=
    else
        echo_c=
        echo_n='-n'
    fi

    # - escape code
    local __r=`echo -e "\x1B[1mX\x1B[0m"`
    echo $__r | grep x1B > /dev/null
    if [ $? -eq 0 ]; then
        e=e
    else
        e=x1B
    fi

    # Resolve options for 'ping'

    # Resolve if sleep accepts only integers
    intsec=0 ; # Accepts not integers as well
    intusec=0 ; # usleep available
    sleep 0.01 2>&-
    if [ $? -ne 0 ]; then
        intsec=1 ; # Accepts only integers
    fi
    usleep 10 2>&-
    if [ $? -eq 0 ]; then
        intusec=1 ; # usleep accepts integers (microseconds)
    fi

    dateif=""
    epochf=""
    # Resolve EPOCH time format for GNU date
    date -u -d '1970-01-01 00:00:00' 2>&- > /dev/null
    if [ $? -eq 0 ]; then
        date -u -d '@100000000' 2>&- > /dev/null
        if [ $? -eq 0 ]; then
            epochf="@"
        else
            epochf="00:00:"
        fi
    fi

    # Portable commands and options
    case "$__sname" in
        AIX )
            ;;        
        DEC )
            ;;        
        HPU )
            bash=/usr/bin/bash
            df=/usr/bin/bdf
            ping=/etc/ping
            remote=/usr/bin/remsh
            tar=/usr/bin/tar
            xterm=/usr/bin/X11/xterm
            date_date="-u"
            df_k=""
            echo_escape=-e
            echo_newline=-n
            ls_ignore=""
            ls_nobackup=""
            ping_wait=""
            ping_count=""
            ping_hpcount="-n 1"
            count=0
            read_timeout=""
            stty_size="size"
            uname_machine=-m
            id_opts=-un
            ;;        
        LIN )
            bash=/bin/bash
            df=/bin/df
            ping=/bin/ping
            remote=/usr/bin/rsh
            tar=/bin/tar
            xterm=/usr/bin/xterm
            date_date="-u -d"
            df_k="-k"
            echo_escape=-e
            echo_newline=-n
            ls_ignore='--ignore=.svn'
            ls_nobackup="B"
            ping_wait="-W 10"
            ping_count="-c 1"
            ping_hpcount=""
            count=1000
            read_timeout="-t 60"
            stty_size="size"
            uname_machine=-m
            id_opts=-un
            ;;        
        MAC )
            bash=/bin/bash
            df=/bin/df
            ping=/sbin/ping
            remote=/usr/bin/rsh
            tar=/usr/bin/tar
            xterm=/usr/X11/bin/xterm
            date_date="-u -j -f "
            df_k="-k"
            dateif="$DF"
            epochf="%s "
            echo_escape=-e
            echo_newline=-n
            ls_ignore=""
            ls_nobackup=""
            ping_wait="-W 10"
            ping_count="-c 1"
            ping_hpcount=""
            count=0
            read_timeout="-t 60"
            stty_size="size"
            uname_machine=-m
            id_opts=-un
            ;;        
        OSF )
            ;;        
        SCO )
            ;;        
        SGI )
            ;;        
        SUNBSD )
            ;;        
        SOLARIS )
            bash=/usr/bin/bash
            df=/usr/sbin/df
            ping=/usr/sbin/ping
            remote=/usr/bin/rsh
            tar=/usr/sbin/tar
            xterm=/usr/openwin/bin/xterm
            date_date="-u"
            df_k="-k"
            echo_escape=-e
            echo_newline=-n
            ls_ignore=""
            ls_nobackup=""
            ping_wait=""
            ping_count=""
            ping_hpcount="56 1"
            count=0
            read_timeout=""
            stty_size=""
            uname_machine=-p
            id_opts=-p
            ;;        
        SYSTEMV )
            ;;        
        (*)
            bash=bash
            df=df
            ping=ping
            remote=remsh
            tar=tar
            xterm=xterm
            date_date=""
            df_k=""
            echo_escape=""
            echo_newline=""
            ls_ignore=""
            ls_nobackup=""
            ping_wait=""
            ping_count=""
            ping_hpcount=""
            count=0
            read_timeout=""
            stty_size=""
            uname_machine=""
            id_opts=""
            ;;
    esac

    # Resolve options for 'tar'
    echo __test > __test; $tar cf __test.tar __test --exclude='.svn*' 2>/dev/null
    if [ $? -ne 0 ]; then
        tar_exclude=""
    else
        tar_exclude=--exclude
    fi
    rm -f __test __test.tar 2>&-
}

#
# SignalHandlers: Installs signal handlers.
#
SignalHandlers() {
    # Parameters: none
    # Globals:    none
    # Assigns:    nothing
    # Returns:    nothing
    trap 'echo ; Exit 1' SIGINT
    trap 'echo ; Exit 2' SIGHUP SIGQUIT SIGTERM
    # trap 'echo ; Exit $? ${LINENO}' ERR
    trap '' SIGTSTP ; # Ctrl-Z disabled
}

#********#
# Timing #
#********#
#
# Sleep: Suspends the execution for a number of seconds.
#
Sleep() {
    # Parameters:
    # 1: Seconds to sleep
    # Globals:
    # - intsec: Flag indicating if sleep accepts only integer seconds
    # - intusec: Flag indicating if usleep accepts integer microseconds (10^-6)
    # Assigns:    nothing
    # Returns:    nothing
    local __s=${1##*[!+.0-9]*} ; # Matches +1.2 but does not -1.2
    __s=${__s:-1} ; # Default 1 second
    if [ $intusec -eq 1 ]; then
        # Convert a possibly floating point value to integer x 10^6 + decimal
        local __i=${__s%%.*} ; # integer part
        local __p=`expr index $__s .` ; # position of decimal dot
        if [ $__p -gt 0 ]; then
            local __d=${__s##*.} ; # decimal part
            __d=${__d:0:6} ; # take only to 1/10^6 of a second
            local __l=${#__d}
        else
            local __d=0
            local __l=0
        fi
        local __e=$((6 - $__l)) ; # Exponent for the power of 10
        __s=$((__i * 10**6 + __d * 10**$__e)) ; # Total in integer microseconds
        usleep $__s
        return
    fi
    if [ $intsec -eq 1 ]; then
        __s=${__s%%.*} ; # Strips the . and the decimal part
        if [ $__s -eq 0 ]; then
            __s=1
        fi
    fi
    sleep $__s
}

#
# Spinner: Prints a spinning character at the current location of the cursor.
#
Spinner() {
    # Parameters: none
    # Globals:    none
    # Assigns:    nothing
    # Returns:    nothing
    local __stop=0
    trap '' SIGTSTP
    trap '__stop=1' SIGHUP SIGTERM SIGINT
    if [ -t 1 ]; then
        # Have a terminal, use it
        printf -- '\033[s-\b' ; Sleep 0.2 ; # Save cursor position
        while : ; do
            printf    '\033[u/\b' ; Sleep 0.2 ; # Restore cursor position
            printf -- '\033[u-\b' ; Sleep 0.2
            printf    '\033[u\\\b'; Sleep 0.2
            printf    '\033[u|\b' ; Sleep 0.2
            if [ $__stop -eq 1 ]; then
                printf    ' \b'
                break
            fi
        done
    else
        # No terminal, just sleep
        while : ; do
            Sleep 1
            if [ $__stop -eq 1 ]; then
                break
            fi
        done
    fi
}

#
# Waitfor: Prints a message and displays a spinner while waiting for a
#          command to finish.
#
Waitfor() {
    # Parameters:
    # 1: Message to display before the spinner
    # 2: Command to execute and wait for. Each line must end with semi-colon ';'
    # 3: Indentation level after the margin
    # 4: Optional path to log file
    # Globals:    
    # - PID: process id array
    # - SLOT: data popped from the top of the PID stack
    # Assigns:    nothing
    # Returns:    nothing
    local __msg=${1:-"..."}
    local __cmd="${2:-:}"
    local __ind=${3:-0}
    local __log="$4"
    local __datestamp=`date +${DF}`
    local __timestamp=`date +${TF}`
    local __datetime=${__datestamp}"T"${__timestamp}
    local __status=0
    local __ti=0
    local __tf=0
    local __td=0
    local __tl=0
    Debug "cmd = >$__cmd<"
    Tell "Waiting for $__msg\c" $__ind
    stty igncr
    stty -echo
    Spinner &
    Push PID $!

    # Ignore ^C, ^Z
    trap '' SIGINT SIGTERM SIGTSTP
    # Redirect command errors to local file
    rm -f /tmp/__cmd_error__ 2>/dev/null
    echo "CMD in $progname by ${user}@${ws} on $__datetime" > /tmp/__cmd_error__
    chmod 666 /tmp/__cmd_error__ 2>/dev/null
    __cmd=${__cmd//;/2>>/tmp/__cmd_error__;}
    # Execute command and wait until it (hopefuly) finishes
    __ti=${SECONDS}
    eval "$__cmd"
    __status=$?
    __tf=${SECONDS}
    __td=$(($__tf - $__ti))
    # __tl=`date $date_date ${epochf}$(($__tf - $__ti)) +'%T'`
    # More portable conversion
    __tl=$(printf "%02d:%02d:%02d" $(($__td / 3600)) $(($__td / 60)) $(($__td % 60)))

    # Kill Spinner process
    Pop PID
    kill -s SIGTERM ${SLOT} 1>&- 2>&-
    wait ${SLOT}

    # Report command result
    test -n "$__log" && cat /tmp/__cmd_error__ >> "$__log"
    if [ $__status -ne 0 ]; then
        local __line=""
        printf 'Failure! after '$__tl
        Error "Code $__status: ${__msg//./} command failed."
        for __line in "$(awk 'BEGIN{FS="\n"}{print $1}' < /tmp/__cmd_error__ | tail -20 )"; do
            Explain "$__line"
        done
    else
        printf 'Done! in '$__tl
    fi
    rm -f /tmp/__cmd_error__ 2>/dev/null
    stty -igncr
    stty echo
    SignalHandlers
    return $__status
}

#**********#
# Calendar #
#**********#
#
# Period: Calculates which observation period a date belongs to.
#
Period() {
    # Parameters:
    # 1: date (optional)
    # Globals:
    # - DF: date format
    # Assigns:    nothing
    # Returns:
    # - period: positive integer for dates after 1968-APR-01
    local __d
    local __period=0
    local __date=${1:-`date +${DF}`}
    # Test if parameter has a valid date format
    __d=`date $date_date $dateif "$__date" +'%Y-%m' 2>&-`
    if [ $? -eq 0 ]; then
        # Year with century
        local __year=`date $date_date $dateif "$__date" +'%Y'`
        # Month without leading zeroes
        local __month=`date $date_date $dateif "$__date" +'%m' | awk '{printf("%d",$1)}'`
        __period=$((((${__year}-1968)*12+${__month}+2)/6))
    fi
    echo $__period
    return $__period
}

#
# LastDay: Returns the number of the last day of each month.
#
LastDay() {
    # Parameters:
    # 1: number of the month
    # 2: number of the year
    # Globals:    none
    # Assigns:    nothing
    # Returns:
    # - lastday: number of the last day of the month for the year
    local __lastday=0
    local __month=${1##*[!0-9]*}
    local __year=${2##*[!0-9]*}
    __month=${__month:-0}
    __year=${__year:-0}
    case $_month in
        01 |  1)
            __lastday=31
            ;;
        02 |  2)
            if [ $(($__year % 4)) -eq 0 ]; then
                __lastday=29
            else
                __lastday=28
            fi
            ;;
        03 |  3)
            __lastday=31
            ;;
        04 |  4)
            __lastday=30
            ;;
        05 |  5)
            __lastday=31
            ;;
        06 |  6)
            __lastday=30
            ;;
        07 |  7)
            __lastday=31
            ;;
        08 |  8)
            __lastday=31
            ;;
        09 |  9)
            __lastday=30
            ;;
        10 | 10)
            __lastday=31
            ;;
        11 | 11)
            __lastday=30
            ;;
        12 | 12)
            __lastday=31
            ;;
    esac
    echo $__lastday
    return $__lastday
}

#
# Yesterday: Returns yesterdays date in YYYY-MM-DD format
#            yesterday=`TZ=aaa24 date +%d-%m-%Y`
#
Yesterday() {
    # Parameters:
    # 1: date (optional)
    # Globals:
    # - DF: date output format
    # - date_date: date option
    # - dateif: date input format
    # Assigns:    nothing
    # Returns:
    # - string yyyy-mm-dd with yesterday's date
    # - 0 (zero) if invalid parameter date
    local __d
    local __date=${1:-`date +$DF`}
    # Test if parameter has a valid date format
    __d=`date $date_date $dateif "$__date" +'%Y-%m-%d' 2>&-`
    if [ $? -eq 0 ]; then
        local __year=`date $date_date $dateif "$__date" +'%Y'`
        local __month=`date $date_date $dateif "$__date" +'%m'`
        local __day=`date $date_date $dateif "$__date" +'%e'`
        __day=$(($__day-1))
        if [ $__day -eq 0 ]; then
            __month=$(($__month-1))
            if [ $__month -eq 0 ]; then
                __year=$(($__year-1))
                __month=12
            fi
            __day=$(LastDay $__month $__year)
            # __day=$?
        fi
        echo `date $date_date $dateif "$__year-$__month-$__day" +${DF}`
    else
        echo 0
    fi
}

#*****#
# Log #
#*****#
#
# StartLog: Prepares log file to receive output from the program.
#
StartLog() {
    # Parameters:
    # 1: Pathname of log file
    # Globals:
    # - log: log flag set with SHELL_LOG environment variable
    # - lflag: log flag set with -l option
    # - dbg: debug flag set with SHELL_DEBUG environment variable
    # - dflag: debug flag set with -d option
    # - LOGFILE: Path to log file
    # Assigns:
    # - lflag: log flag set to FALSE if lfile cannot be used
    # - lfile: Current log file path
    # Returns:    nothing
    if [ $((lflag || log)) -ne 1 ]; then
        return
    fi
    local __path=${1:-$LOGFILE}
    local __logpath=$(dirname  $__path)
    local __logfile=$(basename $__path)
    local __status=0

    if [ -z "$__logpath" -o "$__logpath" = "." ]; then
        __logpath=${PWD}
    else
        if (cd $__logpath >/dev/null 2>&1) && test $? -eq 0; then
            cd $__logpath >/dev/null 2>&1
            __logpath=$PWD
            cd - >/dev/null 2>&1
        fi
    fi
    __logfile=${__logpath}/${__logfile}
    if [ -f $__logfile ]; then
        test -w $__logfile
        __status=$?
    else
        touch $__logfile 2>&-
        __status=$?
        rm -f $__logfile 2>&-
    fi
    if [ $__status -ne 0 ]; then
        lflag=0
        Warning "Log File turned OFF."
        Explain "Cannot open file $__logfile for output."
        if [ $log -eq 1 -a "$__logfile" != "$LOGFILE" ]; then
            Explain "Still logging to default file $LOGFILE"
            Explain "because SHELL_LOG is $SHELL_LOG"
        else
            log=0
        fi
    else
        lfile="$__logfile"
    fi
}

#******#
# Help #
#******#
#
# HelpAd: Prints a message explaining how to get detailed instructions
#         on how to run this program.
HelpAd() {
    # Parameters: none
    # Globals:
    # - progname: Name of the program
    # Assigns:    nothing
    # Returns:    nothing
    Filler
    Explain " To see the Help Screen use "
    Explain " $progname -h"
    Filler
}

#
# Help: Prints detailed instructions on how to run the program.
#
Help() {
    # Parameters:
    # 1: Tag indicating the type of information: help, usage, license
    # 2: List of help lines separated by $LIFS
    # Globals:
    # - progname: Name of the program
    # Assigns:    nothing
    # Returns:    nothing
    local __tag=${1:-Help}
    Frame
    Banner " $__tag: "
    Banner " $progname " 1
    Items "$2"
    Frame
}

#
# Items: Prints the list of accepted values of a parameter.
#
Items() {
    # Parameters:
    # 1: List of item values separated by $LIFS
    # Globals:
    # - LIFS: Local Internal Field Separator
    # - FGCOLOR: Foreground color
    # Assigns:    nothing
    # Returns:    nothing
    local __item
    local __ifs=$IFS
    IFS=$LIFS
    set -- $1
    for __item in $@; do
        Banner "$__item" 2 CURRENT
    done
    IFS=$__ifs
}

#*********#
# Version #
#*********#
#
# Version: Prints the program's revision, date, and author if defined.
#
Version() {
    # Parameters: none
    # Globals:
    # - progname: 
    # Assigns:    nothing
    # Returns:    nothing

    # Hide the pattern so the next grep won't find it here.
    local __pattern=`echo "" | awk '{printf("%c%c%c%c", "@", "(", "#", ")")}'`
    local __path=`which $progname`
    if [ -z "$__path" ]; then
        __path=$progname
    fi
    local __prog=`grep -e "$__pattern" $__path 2>&- | awk '{printf("%s",$4)}'`
    local __revi=`grep -e "$__pattern" $__path 2>&- | awk '{printf("%s",$5)}'`
    local __date=`grep -e "$__pattern" $__path 2>&- | awk '{printf("%s",$6)}'`
    local __auth=`grep -e "$__pattern" $__path 2>&- | awk '{printf("%s",$8)}'`
    __revi=${__revi##*[!0-9]*}
    __revi=${__revi:-000000}
    __date=${__date##*[!-/0-9]*}
    __date=${__date:-0000-00-00}
    __auth=${__auth:-unknown}
    local __pv="$__prog $__revi $__date by $__auth"
    echo "$__pv"
}

#*********#
# Options #
#*********#
#
# GetOpt: Emulates pure POSIX getopt for all OS flavors.
#
# pure-getopt, a drop-in replacement for GNU getopt in pure Bash.
# version 1.4
#
# Copyright 2012 Aron Griffis <aron@arongriffis.com>
# Released under the GNU GPL v3
# Email me to request another license if needed for your project.
#
GetOpt() {
    # Usage:
    #  getopt optstring parameters
    #  getopt [options] [--] optstring parameters
    #  getopt [options] -o|--options optstring [options] [--] parameters
    # 
    # Options:
    #  -a, --alternative Allow long options starting with single -
    #  -h, --help This small usage guide
    #  -l, --longoptions <longopts> Long options to be recognized
    #  -n, --name <progname> The name under which errors are reported
    #  -o, --options <optstring> Short options to be recognized
    #  -q, --quiet Disable error reporting by getopt(3)
    #  -Q, --quiet-output No normal output
    #  -s, --shell <shell> Set shell quoting conventions
    #  -T, --test Test for getopt(1) version
    #  -u, --unquote Do not quote the output
    #  -V, --version Output version information
    
    _getopt_main() {
        # Returns one of the following statuses:
        # 0 success
        # 1 error parsing parameters
        # 2 error in getopt invocation
        # 3 internal error
        # 4 reserved for -T
        #
        # For statuses 0 and 1, generates normalized and shell-quoted
        # "options -- parameters" on stdout.
        declare parsed status
        declare short long name flags
        declare have_short=false

        # Synopsis from getopt man-page:
        #
        # getopt optstring parameters
        # getopt [options] [--] optstring parameters
        # getopt [options] -o|--options optstring [options] [--] parameters
        #
        # The first form can be normalized to the third form which
        # _getopt_parse() understands. The second form can be recognized after
        # first parse when $short hasn't been set.

        if [[ -n ${GETOPT_COMPATIBLE+isset} || $1 == [^-]* ]]; then
            # Enable compatibility mode
            flags=c$flags
            # Normalize first to third synopsis form
            set -- -o "$1" -- "${@:2}"
        fi

        # First parse always uses flags=p since getopt always parses its own
        # arguments effectively in this mode.
        parsed=$(_getopt_parse getopt ahl:n:o:qQs:TuV \
          alternative,help,longoptions:,name,options:,quiet,quiet-output,shell:,test,version \
          p "$@")
        status=$?
        if [[ $status != 0 ]]; then
            if [[ $status == 1 ]]; then
                echo "Try \`getopt --help' for more information." >&2
                # Since this is the first parse, convert status 1 to 2
                status=2
            fi
            return $status
        fi
        eval "set -- $parsed"

        while [[ $# -gt 0 ]]; do
            case $1 in
                (-a|--alternative)
                    flags=a$flags
                    ;;
                (-h|--help)
                    _getopt_help
                    return 2 # as does GNU getopt
                    ;;
                (-l|--longoptions)
                    long="$long${long:+,}$2"
                    shift
                    ;;
                (-n|--name)
                    name=$2
                    shift
                    ;;
                (-o|--options)
                    short=$2
                    have_short=true
                    shift
                    ;;
                (-q|--quiet)
                    flags=q$flags
                    ;;
                (-Q|--quiet-output)
                    flags=Q$flags
                    ;;
                (-s|--shell)
                    case $2 in
                        (sh|bash)
                            flags=${flags//t/}
                            ;;
                        (csh|tcsh)
                            flags=t$flags
                            ;;
                        (*)
                            echo 'getopt: unknown shell after -s or --shell argument' >&2
                            echo "Try \`getopt --help' for more information." >&2
                            return 2
                            ;;
                    esac
                    shift
                    ;;
                (-u|--unquoted)
                    flags=u$flags
                    ;;
                (-T|--test)
                    echo `getopt -V 2>&-`
                    return 4
                    ;;
                (-V|--version)
                    echo "pure-getopt 1.4"
                    return 0
                    ;;
                (--)
                    shift
                    break
                    ;;
            esac
            shift
        done

        if ! $have_short; then
            # $short was declared but never set, not even to an empty string.
            # This implies the second form in the synopsis.
            if [[ $# == 0 ]]; then
                echo 'getopt: missing optstring argument' >&2
                echo "Try \`getopt --help' for more information." >&2
                return 2
            fi
            short=$1
            have_short=true
            shift
        fi

        if [[ $short == -* ]]; then
            # Leading dash means generate output in place rather than reordering,
            # unless we're already in compatibility mode.
            [[ $flags == *c* ]] || flags=i$flgas
            short=${short#?}
        elif [[ $short == +* ]]; then
            # Leading plus means POSIXLY_CORRECT, unless we're already in
            # compatibility mode.
            [[ $flags == *c* ]] || flags=p$flags
            short=${short#?}
        fi

        # This should fire if POSIXLY_CORRECT is in the environment, even if
        # it's an empty string. That's the difference between :+ and +
        flags=${POSIXLY_CORRECT+p}$flags

        _getopt_parse "${name:-getopt}" "$short" "$long" "$flags" "$@"
    }

    _getopt_parse() {
        # Inner getopt parser, used for both first parse and second parse.
        # Returns 0 for success, 1 for error parsing, 3 for internal error.
        # In the case of status 1, still generates stdout with whatever could
        # be parsed.
        #
        # $flags is a string of characters with the following meanings:
        # a - alternative parsing mode
        # c - GETOPT_COMPATIBLE
        # i - generate output in place rather than reordering
        # p - POSIXLY_CORRECT
        # q - disable error reporting
        # Q - disable normal output
        # t - quote for csh/tcsh
        # u - unquoted output

        declare name="$1" short="$2" long="$3" flags="$4"
        shift 4

        # Split $long on commas, prepend double-dashes, strip colons;
        # for use with _getopt_resolve_abbrev
        declare -a longarr
        _getopt_split longarr "$long"
        longarr=( "${longarr[@]/#/--}" )
        longarr=( "${longarr[@]%:}" )
        longarr=( "${longarr[@]%:}" )

        # Parse and collect options and parameters
        declare -a opts params
        declare o alt_recycled=false error=0

        while [[ $# -gt 0 ]]; do
            case $1 in
                (--)
                    params=( "${params[@]}" "${@:2}" )
                    break
                    ;;
                (--*=*)
                    o=${1%%=*}
                    if ! o=$(_getopt_resolve_abbrev "$o" "${longarr[@]}"); then
                        error=1
                    elif [[ ,"$long", == *,"${o#--}"::,* ]]; then
                        opts=( "${opts[@]}" "$o" "${1#*=}" )
                    elif [[ ,"$long", == *,"${o#--}":,* ]]; then
                        opts=( "${opts[@]}" "$o" "${1#*=}" )
                    elif [[ ,"$long", == *,"${o#--}",* ]]; then
                        if $alt_recycled; then
                            # GNU getopt isn't self-consistent about whether it reports
                            # errors with a single dash or double dash in alternative
                            # mode, but in this case it reports with a single dash.
                            _getopt_err "$name: option '${o#-}' doesn't allow an argument"
                        else
                            _getopt_err "$name: option '$o' doesn't allow an argument"
                        fi
                        error=1
                    else
                        echo "getopt: assertion failed (1)" >&2
                        return 3
                    fi
                    alt_recycled=false
                    ;;
                (--?*)
                    o=$1
                    if ! o=$(_getopt_resolve_abbrev "$o" "${longarr[@]}"); then
                        error=1
                    elif [[ ,"$long", == *,"${o#--}",* ]]; then
                        opts=( "${opts[@]}" "$o" )
                    elif [[ ,"$long", == *,"${o#--}::",* ]]; then
                        opts=( "${opts[@]}" "$o" '' )
                    elif [[ ,"$long", == *,"${o#--}:",* ]]; then
                        if [[ $# -ge 2 ]]; then
                            shift
                            opts=( "${opts[@]}" "$o" "$1" )
                        else
                            _getopt_err "$name: option '$o' requires an argument"
                            error=1
                        fi
                    else
                        echo "getopt: assertion failed (2)" >&2
                        return 3
                    fi
                    alt_recycled=false
                    ;;
                (-*)
                    if [[ $flags == *a* ]]; then
                        # Alternative parsing mode!
                        # Try to handle as a long option if any of the following apply:
                        # 1. There's an equals sign in the mix -x=3 or -xy=3
                        # 2. There's 2+ letters and an abbreviated long match -xy
                        # 3. There's a single letter and an exact long match
                        # 4. There's a single letter and no short match
                        o=${1::2} # temp for testing #4
                        if [[ $1 == *=* || $1 == -?? || \
                              ,$long, == *,"${1#-}"[:,]* || \
                              ,$short, != *,"${o#-}"[:,]* ]]; then
                            o=$(_getopt_resolve_abbrev "${1%%=*}" "${longarr[@]}" 2>/dev/null)
                            case $? in
                              (0)
                                  # Unambiguous match. Let the long options parser handle
                                  # it, with a flag to get the right error message.
                                  set -- "-$@"
                                  alt_recycled=true
                                  continue
                                  ;;
                              (1)
                                  # Ambiguous match, generate error and continue.
                                  _getopt_resolve_abbrev "${1%%=*}" "${longarr[@]}" >/dev/null
                                  error=1
                                  shift
                                  continue
                                  ;;
                              (2)
                                  # No match, fall through to single-character check.
                                  true
                                  ;;
                              (*)
                                  echo "getopt: assertion failed (3)" >&2
                                  return 3
                                  ;;
                            esac
                        fi
                    fi

                    o=${1::2}
                    if [[ "$short" == *"${o#-}"::* ]]; then
                        if [[ ${#1} -gt 2 ]]; then
                            opts=( "${opts[@]}" "$o" "${1:2}" )
                        else
                            opts=( "${opts[@]}" "$o" '' )
                        fi
                    elif [[ "$short" == *"${o#-}":* ]]; then
                        if [[ ${#1} -gt 2 ]]; then
                            opts=( "${opts[@]}" "$o" "${1:2}" )
                        elif [[ $# -ge 2 ]]; then
                            shift
                            opts=( "${opts[@]}" "$o" "$1" )
                        else
                            _getopt_err "$name: option requires an argument -- '${o#-}'"
                            error=1
                        fi
                    elif [[ "$short" == *"${o#-}"* ]]; then
                        opts=( "${opts[@]}" "$o" )
                        if [[ ${#1} -gt 2 ]]; then
                            set -- "$o" "-${1:2}" "${@:2}"
                        fi
                    else
                        if [[ $flags == *a* ]]; then
                            # Alternative parsing mode! Report on the entire failed
                            # option. GNU includes =value but we omit it for sanity with
                            # very long values.
                            _getopt_err "$name: unrecognized option '${1%%=*}'"
                        else
                            _getopt_err "$name: invalid option -- '${o#-}'"
                            if [[ ${#1} -gt 2 ]]; then
                                set -- "$o" "-${1:2}" "${@:2}"
                            fi
                        fi
                        error=1
                    fi
                    ;;
                (*)
                    # GNU getopt in-place mode (leading dash on short options)
                    # overrides POSIXLY_CORRECT
                    if [[ $flags == *i* ]]; then
                        opts=( "${opts[@]}" "$1" )
                    elif [[ $flags == *p* ]]; then
                        params=( "${params[@]}" "$@" )
                        break
                    else
                        params=( "${params[@]}" "$1" )
                    fi
                    ;;
            esac
            shift
        done

        if [[ $flags == *Q* ]]; then
            true # generate no output
        else
            echo -n ' '
            if [[ $flags == *[cu]* ]]; then
                printf '%s -- %s' "${opts[*]}" "${params[*]}"
            else
                if [[ $flags == *t* ]]; then
                    _getopt_quote_csh "${opts[@]}" -- "${params[@]}"
                else
                    _getopt_quote "${opts[@]}" -- "${params[@]}"
                fi
            fi
            echo
        fi
        return $error
    }

    _getopt_err() {
        if [[ $flags != *q* ]]; then
            printf '%s\n' "$1" >&2
        fi
    }

    _getopt_resolve_abbrev() {
        # Resolves an abbrevation from a list of possibilities.
        # If the abbreviation is unambiguous, echoes the expansion on stdout
        # and returns 0. If the abbreviation is ambiguous, prints a message on
        # stderr and returns 1. (For first parse this should convert to exit
        # status 2.) If there is no match at all, prints a message on stderr
        # and returns 2.
        declare a q="$1"
        declare -a matches
        shift
        for a; do
            if [[ $q == "$a" ]]; then
                # Exact match. Squash any other partial matches.
                matches=( "$a" )
                break
            elif [[ $flags == *a* && $q == -[^-]* && $a == -"$q" ]]; then
                # Exact alternative match. Squash any other partial matches.
                matches=( "$a" )
                break
            elif [[ $a == "$q"* ]]; then
                # Abbreviated match.
                matches=( "${matches[@]}" "$a" )
            elif [[ $flags == *a* && $q == -[^-]* && $a == -"$q"* ]]; then
                # Abbreviated alternative match.
                matches=( "${matches[@]}" "$a" )
            fi
        done
        case ${#matches[@]} in
            (0)
                [[ $flags == *q* ]] || \
                printf "$name: unrecognized option %s\n" >&2 \
                  "$(_getopt_quote "$q")"
                return 2
                ;;
            (1)
                printf '%s' "$matches"; return 0
                ;;
            (*)
                [[ $flags == *q* ]] || \
                printf "$name: option %s is ambiguous; possibilities: %s\n" >&2 \
                  "$(_getopt_quote "$q")" "$(_getopt_quote "${matches[@]}")"
                return 1
                ;;
        esac
    }

    _getopt_split() {
        # Splits $2 at commas to build array specified by $1
        declare IFS=,
        eval "$1=( \$2 )"
    }

    _getopt_quote() {
        # Quotes arguments with single quotes, escaping inner single quotes
        declare s space q=\'
        for s; do
            printf "$space'%s'" "${s//$q/$q\\$q$q}"
            space=' '
        done
    }

    _getopt_quote_csh() {
        # Quotes arguments with single quotes, escaping inner single quotes,
        # bangs, backslashes and newlines
        declare s i c space
        for s; do
            echo -n "$space'"
            # This syntax of for is not available in older HP-UX
            # for (( i=0; i<${#s}; i++ )) ; do
            i=0
            while [ $i -lt ${#s} ]; do
                c=${s:i:1}
                case $c in
                    (\\|\'|!)
                        echo -n "'\\$c'"
                        ;;
                    ($'\n')
                        echo -n "\\$c"
                        ;;
                    (*)
                        echo -n "$c"
                        ;;
                esac
                i=$((i + 1))
            done
            echo -n \'
            space=' '
        done
    }

    _getopt_help() {
        return
    }

    _getopt_main "$@"
    declare status=$?
    unset -f _getopt_main \
             _getopt_err \
             _getopt_parse \
             _getopt_quote \
             _getopt_quote_csh \
             _getopt_resolve_abbrev \
             _getopt_split _getopt_help
    return $status
}

#************#
# Parameters #
#************#
#
# ExpandRange: Transforms a range into a sequence.
#
ExpandRange() {
    # Parameters:
    # 1: Sequence or Range to expand: [i|i,j,k,...|i..m]
    # Globals:
    # - LIT: Array of literals
    # Assigns:    nothing
    # Returns:
    # - Expanded sequence: [i j k ...]
    local __r="$*"
    eval set -- {$__r}
    # If range or sequence expansion fails, $@ comes out within '{' and '}'
    # so we remove those. If expansion succeeded then next has no effect but
    # assigning the sequence as it is. 
    __r="${@//[\}\{]/}"
    if [ ! -z "${__r##*[!\-.,0-9a-zA-Z]*}" ]; then
        # Still '.', ',' and/or '-' present, that means range didn't expand,
        # so we expand it with another method.
        # - Upper limit:
        local __u=`echo ${__r##*[\-.,]}`
        # - Lower limit:
        local __l=`echo ${__r%%[\-.,]*}`
        local __i=$__l
        local __s=""
        if [[ ! -z "${__u##*[!0-9]*}" &&
              ! -z "${__l##*[!0-9]*}" ]]; then
            # For range of positive integers
            local __test=0
            # Remove non significant zeroes from left side
            __test=$(echo "scale = 0; $__l + 0" | bc 2>&-)
            if [ -n "$__test" ]; then
                __l=$__test
            fi
            __test=$(echo "scale = 0; $__u + 0" | bc 2>&-)
            if [ -n "$__test" ]; then
                __u=$__test
            fi
            if [ $__l -le $__u ]; then
                __s=+
            else
                __s=-
            fi
            __r=""
            # This loop generates ascending (+) or descending (-) sequence.
            while [[ $__i -ne $((__u $__s 1)) ]]; do
                __r="$__r$__i "
                __i=$((__i $__s 1))
            done
        else
            # For range of literals, use the LIT array of literals to search
            local __j=0
            while [[ "$__u" != "${LIT[$__j]}" ]]; do __j=$((__j + 1)); done
            __u=$__j
            __j=0
            while [[ "$__l" != "${LIT[$__j]}" ]]; do __j=$((__j + 1)); done
            __l=$__j
            # Both limits are now converted to positive integers which index
            # LIT array of literals.
            if [ $__l -le $__u ]; then
                __s=+
            else
                __s=-
            fi
            __i=$__l
            __r=""
            # This loop generates ascending (+) or descending (-) sequence.
            while [[ $__i -ne $((__u $__s 1)) ]]; do
                __r="$__r${LIT[$__i]} "
                __i=$((__i $__s 1))
            done
        fi
    fi
    # tr is not very portable. Kept for reference.
    # __r=`echo "${__r}" | tr -s '\-.,;][*?}{^' ' '`
    echo "$__r"
}

#
# CheckRange: Checks if a value is within a specific range or not.
#
CheckRange() {
    # Parameters:
    # 1: Value to check
    # 2: Check type
    #    - E : Element of a set
    #    - I : Integer signed 32 bits with lower and upper limits
    #    - I+: Positive integer including zero
    #    - I-: Negative integer
    #    - R : Real number in floating point precission of 9 decimals
    #    - S : No-null string of printable characters
    # 3: Comparison reference:
    #    - Set for E type check, can be a list, a 'i..j' range
    #      or a comma separated set
    #    - Optional lower limit for I and R types
    # 4: Optional upper limit for I and R types
    # Globals:
    # - MININT32: Lower limit of integer arithmetics
    # - MAXINT32: Upper limit of integer arithmetics
    # Assigns:    nothing
    # Returns:
    # -  0: parameter is within range
    # -  1: parameter is outside range to the right (beyond upper limit) or not
    #       a member of the set
    # - -1: parameter is outside range to the left (beyond lower limit)
    # -  9: invalid check type
    # - -9: invalid value to check
    # Usage:
    # reply=$(CheckRange 277 E {200..250},275,277,290)
    local __v=${1:-null}
    local __t=${2}
    local __l="${3}"
    local __u=${4:-${MAXINT32}}
    local __code=0
    local __test=0
    if [[ -z "$__l" ]]; then
        __l=$MININT32
    fi
    if [[ "$__v" == "null" ]]; then
        __code=-9
    else
        case $__t in
            (E)
                # Test if value is an element of a set
                local __item=""
                local __list=""
                __list=$(ExpandRange $__l)
                if [ -z "$__list" ]; then
                    __list=empty
                fi
                __code=1
                for __item in $__list ; do
                    if [[ "$__v" == "$__item" ]]; then
                        __code=0
                        break
                    fi
                done
            ;;
            (I*)
                # Test if value is an integer number
                if [[ -z "${__v##*[!\-+0-9]*}" ]]; then
                    # Value contains other characters than digits and signs,
                    # cannot be a proper integer number.
                    __code=1
                else
                    # Test if it is a proper integer number
                    __test=$(echo "scale = 0; $__v + 0" | bc 2>&-)
                    if [ -z "$__test" ]; then
                        __code=-9
                    else
                        # Remove non significant zeroes from left side
                        __v=$__test
                        # Test the lower and upper limits
                        case $__t in
                            (I-)
                                __code=1
                                if [[ $__v -lt 0 ]]; then
                                    __code=0
                                fi
                            ;;
                            (I+)
                                __code=-1
                                if [[ $__v -ge 0 ]]; then
                                    __code=0
                                fi
                            ;;
                            (I)
                                if [[ $__v -gt $__u ]]; then
                                    __code=1
                                fi
                                if [[ $__v -lt $__l ]]; then
                                    __code=-1
                                fi
                            ;;
                        esac
                    fi
                fi
            ;;
            (R)
                # Test if value is a real number
                if [[ -z "${__v##*[!\-+.0-9]*}" ]]; then
                    # Value contains other characters than digits, point and
                    # signs, cannot be a proper real number.
                    __code=1
                else
                    __test=$(echo "scale = 10; $__v + 0.0" | bc 2>&-)
                    if [ -z "$__test" ]; then
                        __code=-9
                    else
                        # Remove non significant zeroes from left side
                        __v=$__test
                        if [[ $(echo "scale=10; $__v > $__u" | bc 2>&-) -eq 1 ]] ; then
                            __code=1
                        fi
                        if [[ $(echo "scale=10; $__v < $__l" | bc 2>&-) -eq 1 ]] ; then
                            __code=-1
                        fi
                    fi
                fi
            ;;
            (S)
                # Test if value is a printable string by replacing every
                # printable character with a '#'
                __test=`echo -e "$__v" | sed 's/[[:print:]]/#/g'`
                if [[ ! -z "${__test##*[!#]*}" ]]; then
                    # It contained only '#'s so the string is printable
                    __code=0
                else
                    # It contained something else besides '#' so the string
                    # contains non printable characters.
                    __code=1
                fi
            ;;
            (--)
                __code=9
            ;;
        esac
    fi
    echo $__code
    return $__code
}

#
# ParNum: Prints an error message about the number of parameters received.
#
ParNum() {
    # Parameters:
    # 1: Number of parameters received
    # Globals:
    # - EC_REJECT: Exit code of rejection
    # Assigns:
    # - eflag: Error flag TRUE
    # Returns:    nothing
    eflag=$EC_REJECT
    Error "Number of parameters is incorrect: $1"
}

#
# ParVal: Prints detailed message about the reasons why a parameter is
#         not accepted.
#
ParVal() {
    # Parameters:
    # 1: Name of parameter
    # 2: Value of parameter
    # 3: Help on parameter values as a list of $LIFS separated strings
    # Globals:    none
    # Assigns:    nothing
    # Returns:    nothing
    if [ -z "$2" ]; then
        Error   "Parameter $1 is empty."
    else
        Error   "Parameter $1 = $2 is incorrect."
    fi
    Explain "$1 value must be: "
    Items "$3"
}

#*************#
# Environment #
#*************#
#
# CheckRepository: Checks that inside a working copy and
#                  access to SVN repository.
#
CheckRepository() {
    # Parameters:
    # 1: Name of assembly working copy
    # 2: URL of repository
    # Globals:    none
    # Assigns:    nothing
    # Returns:
    #  0: All OK, or no assembly, or no SVN environment
    # 10: Error code: SVN Server unavailable
    # 11: Error code: Working copy does not contain required assembly
    local __assembly="${1:-.}"
    local __repo="${2:-''}"
    local __ec=0
    if [ -n "$__assembly" ] ; then
        if [ -n "$__repo" ]; then
            # When in a SVN environment, update revision info
            if [ ! -d .svn ]; then
                Warning "This is not a working copy directory"
                Explain "Make sure you work with a copy of $__assembly from"
                Explain "${__repo%/}/path/to/the/assembly/$__assembly"
            else
                local __wc_rev=UNDEFINED
                local __wc_lch=UNDEFINED
                local __wc_aut=UNDEFINED
                local __wc_url=UNDEFINED
                local __wc_ass=UNDEFINED
                # Check SVN server response
                svn info . 2>&- > dummy
                if [ $? -ne 0 ]; then
                    Error   "SVN Server unavailable"
                    Explain "Check connection to $__repo"
                    __ec=10
                else
                    __wc_rev=`svn info . 2>&- |\
                              awk '/Last Changed Rev:/{printf("%s",$4)}'`
                    __wc_lch=`svn info . 2>&- |\
                              awk '/Last Changed Date:/{printf("%s",$4)}'`
                    __wc_aut=`svn info . 2>&- |\
                              awk '/Last Changed Author:/{printf("%s",$4)}'`
                    __wc_url=`svn info . 2>&- |\
                              awk '/^URL:/{printf("%s",$2)}'`
                    __wc_ass=`echo $__wc_url $__assembly |\
                              awk '{i=index($1,$2); \
                                  if (i > 0) \
                                      printf("%s",substr($1,i)); \
                                  else \
                                      print("UNDEFINED")} \
                                  '`
                    if [ "$__wc_ass" != "$__assembly" ]; then
                        Error "This working copy does not contain assembly $__assembly"
                        Explain "Make sure you work with a copy of $__assembly"
                        __ec=11
                    fi
                fi
                rm -f dummy
                Frame
                Banner "Working copy of $__assembly: "
                Banner "  REV: $__wc_rev on $__wc_lch by $__wc_aut "
                Banner "  URL: $__wc_url "
                Frame
            fi
        else
            Warning "This is not a SVN environment"
            Explain "Make sure you work with an updated"
            Explain "copy of the $__assembly software"
        fi
    else
        Tell "No assembly defined."    
    fi
    return $__ec
}

#***********#
# Variables #
#***********#
#
# Delete: Deletes an external variable's file.
#
Delete() {
    # Parameters:
    # 1: Variable name
    # Globals:
    # - hd: home directory
    # Assigns:    nothing
    # Returns:    nothing
    local __varname=${1// /_}
    if [ -f $hd/._$__varname.var ]; then
        rm -f $hd/._$__varname.var
    fi
}

#
# Export: Saves the external variable's value in a file.
#
Export() {
    # Parameters:
    # 1: Variable name
    # 2: Variable value
    # Globals:
    # - hd: home directory
    # Assigns:    nothing
    # Returns:    nothing
    local __varname=${1// /_}
    shift
    local __varvalue="$*"
    echo "$__varvalue" > $hd/._$__varname.var
}

#
# Import: Opens the external variable's file and returns its contents.
#
Import() {
    # Parameters:
    # 1: Variable name
    # Globals:
    # - hd: home directory
    # Assigns:    nothing
    # Returns:
    # - Variable value
    local __varname=${1// /_}
    local __varvalue=""
    if [ -f $hd/._$__varname.var ]; then
        __varvalue=`cat $hd/._$__varname.var`
    fi
    echo "$__varvalue"
}

#********#
# Stacks #
#********#
#
# Stack: Returns the current size of the stack, which is the available slot for
#        next push.
#
Stack() {
    # Parameters:
    # 1: Name of the stack
    # Globals:    none
    # Assigns:
    # Returns:
    # - 0: Stack empty or does not exist
    # - n: Stack size which corresponds to the next slot for push
    local __top=0
    eval __top=\${#$1[*]}
    echo $__top
    return $__top
}

#
# Pop: Pops the top stack slot.
#
Pop() {
    # Parameters:
    # 1: Name of the stack
    # Globals:    none
    # Assigns:
    # - SLOT: The top slot of the stack indirectly referenced by the name in $1
    # Returns:
    # - 0: All OK, echoes data from the top of the stack
    # - 1: NOK, echoes an empty value
    local __top=$(Stack $1)
    local __data=''
    local __ec=0
    if [ $__top -eq 0 ]; then
        __ec=1
    else
        local __t=$((__top - 1))
        eval __data=\${$1[$__t]}
        unset $1[$__t]
    fi
    SLOT="$__data"
    return $__ec
}

#
# Push: Pushes data to the top of the stack.
#
Push() {
    # Parameters:
    # 1: Name of the stack
    # 2: Data to push into the top
    # Globals:    none
    # Assigns:
    # - The top slot of the stack indirectly referenced by the name in $1
    # Returns:
    # - 0: All OK, echoes data from the top of the stack
    # - 1: NOK, echoes an empty value
    local __top=$(Stack $1)
    local __data="$2"
    local __ec=0
    eval $1[$__top]="\"$__data\""
    __ec=$?
    return $__ec
}

#**********#
# Cleaning #
#**********#
#
# CleanUp: Performs all cleaning actions necessary to minimize 
#          program's footprint.
CleanUp() {
    # Parameters: none
    # Globals:
    # - PID: Background process id array
    # - VAR: External variable array
    # Assigns:    nothing
    # Returns:    nothing

    # Kill child processes if any
    if [ ${#PID[*]} -gt 1 ]; then
        StopProcs
        Doing "Killing background processes..." 1
        KillProcs
        Done "Killing background processes." 1
    fi

    # Delete external variables if any
    if [ ${#VAR[*]} -gt 0 ]; then
        Doing "Cleaning up external variables..."
        local __i=$((${#VAR[*]} - 1))
        while [ $__i -ge 0 ]; do
            Delete ${VAR[$__i]}
            unset VAR[$__i]
            __i=$((__i - 1))
        done
        Done "Cleaning up external variables."
    fi
    return
}

#
# StopProcs: Suspends registered background processes.
#
StopProcs() {
    # Parameters: none
    # Globals:
    # - PID: Background process id array
    # Assigns:    nothing
    # Returns:    nothing
    local __i=$((${#PID[*]} - 1))
    while [ $__i -ge 1 ]; do
        kill -0 ${PID[$__i]} 2>&-
        if [ $? -eq 0 ]; then
            kill -s SIGSTOP ${PID[$__i]} 2>&-
            if [ $? -eq 0 ]; then
                Tell "Stopped ${PID[$__i]}: $__pin" 1
            else
                Warning "Could not stop ${PID[$__i]}"
            fi
        fi
        __i=$((__i - 1))
    done
}

#
# KillProcs: Kills registered background processes.
#
KillProcs() {
    # Parameters: none
    # Globals:
    # - PID: Background process id array
    # Assigns:    nothing
    # Returns:    nothing
    local __i=$((${#PID[*]} - 1))
    local __pin=""
    local __stat=0
    while [ $__i -ge 1 ]; do
        kill -0 ${PID[$__i]} 2>&-
        if [ $? -eq 0 ]; then
            __pin=`ps -fp ${PID[$__i]} |\
                   grep ${PID[$__i]} |\
                   awk '{printf("%s %s",$8,$9)}' \
                  `
            kill -s SIGCONT ${PID[$__i]} 2>&-
            kill -s SIGTERM ${PID[$__i]} 2>&-
            wait ${PID[$__i]}
            __stat=$?
            printf '\r'
            if [ $__stat -eq 0 ]; then
                Tell "Killed ${PID[$__i]}: $__pin" 1
                unset PID[$__i]
            else
                Warning "Could not kill ${PID[$__i]}: $__pin"
            fi
        fi
        __i=$((__i - 1))
    done
}

#********#
# Errors #
#********#
#
# CheckStatus: Prints OK or an error message after checking status code,
#              and sets the error flag.
#
CheckStatus() {
    # Parameters:
    # 1: Status
    #    -  0: Successful
    #    - !0: Failure
    # 2: Status code for failure
    # 3: Line number where the error code was sent
    # 4: Exit code to assign to eflag
    # 5: Error flag to print a message or not
    #    - E: Error
    #    - I: Info
    #    - W: Warning
    #    - X: No print
    # 6: Message to print in case of error flag
    # 7: Indentation level after the margin
    # Globals:
    # - EC_ABORT:
    # - EC_CANCEL:
    # Assigns:
    # - eflag: Error flag TRUE
    # Returns:    nothing

    local __status=$1
    local __scode=$2
    local __lineno=$3
    local __ecode=$4
    local __eflag=$5
    local __emsg="$6"
    local __ilevel=$7
    local __reply=OK
    test $__status -eq 0 || __reply="FAILURE (code $__scode at $__lineno)"
    Tell "Status: $__reply" $__ilevel
    if [ $__status -ne 0 ]; then
        eflag=$__ecode
        case $__eflag in
            E )
                Error "$__emsg"
                ;;
            I )
                Explain "$__emsg"
                ;;
            W )
                Warning "$__emsg"
                ;;
            * )
                ;;
        esac
    fi
}

#******#
# Exit #
#******#
#
# Exit: Standard exit for program.
#
Exit() {
    # Parameters:
    # 1: Exit level
    #    0: Normal end
    #    1: Cancel
    #    2: Abort
    #    3: Reject
    #    4: Informative
    #    9: Normal end and skip cleanup and banner
    #    *: Error code
    # 2: Line number where error occurred
    # Globals:
    # - progname: program name
    # - save_tty: tty original configuration
    # - user:
    # - ws:
    # - lflag: 
    # - log:
    # - DF: Date output format
    # - TF: Time output format
    # - EC_*: Exit codes
    # Assigns:    nothing
    # Returns:
    # - Exit code
    local __datestamp=`date +${DF}`
    local __timestamp=`date +${TF}`
    local __datetime=${__datestamp}"T"${__timestamp}
    local __ec=${1##*[!-0-9]*}
    local __ln=${2##*[!0-9]*}
    __ec=${__ec:-$?}
    __ln=${__ln:-unknown}
    stty $save_tty 2>&-
    if [ "$__ec" -lt $EC_REJECT -o "$__ec" -gt $EC_QUIET ]; then
        CleanUp
    fi
    case $__ec in
        $EC_END)
            Frame
            Banner "END    $progname by ${user}@${ws} on $__datetime"
            Frame
            ;;
        $EC_CANCEL)
            Frame
            Banner "CANCEL $progname by ${user}@${ws} on $__datetime"
            Frame
            ;;
        $EC_ABORT)
            Frame $FG_ERROR
            Banner "ABORT  $progname by ${user}@${ws} on $__datetime" 0 CURR
            Frame CURR
            ;;
        $EC_REJECT)
            Frame $FG_WARN
            Banner "REJECT $progname by ${user}@${ws} on $__datetime" 0 CURR
            Frame CURR
            ;;
        $EC_INFO)
            Frame
            Banner "INFO $progname by ${user}@${ws} on $__datetime"
            Frame
            ;;
        $EC_QUIET)
            if [ $((lflag || log)) -eq 1 ]; then
                Frame
                Banner "END    $progname by ${user}@${ws} on $__datetime"
                Frame
            fi
            __ec=0
            ;;
        (*)
            Frame
            Banner "ERROR  $progname ${user}@${ws} at code $__ec in line $__ln on $__datetime"
            Frame
            ;;
    esac
    printf '\n'
    exit $__ec
}

#*****************#
# Formatted input #
#*****************#
#
# GetAnswer: Receives user input with timeout, default, help, menu.
#
GetAnswer() {
    # Parameters: none
    # 1: Question
    # 2: Default answer
    # 3: Help message
    # 4: Timeout in seconds
    # Globals:
    # - MARGIN: Left margin blank string
    # - read_timeout:
    # Assigns:    nothing
    # Returns:
    # - Answer:
    # - Exit code:
    #   - 0: Normal answer from user
    #   - 1: Default answer after ENTER
    #   - 2: Default answer after timeout
    #   - 3: Empty answer after timeout and no default answer available
    #   - 4: Positive answer (Y) from user
    #   - 5: Negative answer (N) from user
    #   - 6: Quit answer (Q) from user
    local __question=${1:-""}
    local __default=${2:-""}
    local __help=${3:-""}
    local __to=${4##*[!0-9]*}
    local __answer=""
    local __prompt=""
    local __saved=""
    local __menu="ENTER: accept | ?: Help | Y: yes | N: no | Q: quit"
    local __ec=0
    __to=${__to:-0}
    # __question=${__question/%\?/""}
    if [ -z "$__default" ]; then
        __question="$__question --> "
    else
        __question="$__question [$__default]--> "
        __saved="$__default"
    fi
    if [ $__to -gt 0 ]; then
        __to=${read_timeout//60/$__to}
    else
        __to=""
    fi
    __prompt="$__question"
    while : ; do
        read $__to -p "$MARGIN$__prompt " __answer
        if [ $? -ne 0 ]; then
            __answer=timeout
        fi
        case $(echo "$__answer" | tr '[:upper:]' '[:lower:]') in
            "" ) 
                if [ -n "$__default" ]; then
                    __answer="$__default"
                    __ec=1
                    break
                fi
                __prompt="$__question"
                __default="$__saved"
                ;;
            timeout )
                if [ -n "$__default" ]; then
                    __answer="$__default"
                    __ec=2
                else
                    __answer=""
                    __ec=3
                fi
                break
                ;;
            yes | y )
                __answer=Y
                __ec=4
                break
                ;;
            no | n )
                __answer=N
                __ec=5
                break
                ;;
            quit | q )
                __answer=Q
                __ec=6
                break
                ;;
            !* )
                # eval ${__answer#\!}
                __prompt="$__menu"
                __default=""
                ;;
            \? )
                if [ -z "$__help" ]; then
                    __prompt="$__menu"
                else
                    __prompt="$__help"
                fi
                __default=""
                ;;
            * )
                break
                ;;
        esac
    done

    eval echo "$__answer"
    return $__ec
}

#******************#
# Formatted output #
#******************#
#
# PrintLine: Prints a message in one or more justified lines of constant length.
#
PrintLine() {
    # Parameters:
    # 1: Message to print
    # 2: Max. line length. If $1 is longer, it will be chopped and the rest
    #    printed in next line
    # 3: Initial line format
    # 4: Secondary line format in case message gets chopped
    # 5: Optional suffix to be appended to last chunck of $1
    # 6: Optional foreground color code
    # Globals:
    # - MAXCOL: Max number of columns available on terminal
    # - log: log flag set with SHELL_LOG environment variable
    # - lflag: log flag set with -l option
    # - lfile: Current log file
    # - echo_c: echo no_newlines options
    # - echo_n:
    # - echo_escape: echo escape_code_activate option
    # - P1: Patterns to match the echo escape codes
    # - P2:
    # - P3:
    # - P4:
    # - RESET: Text attributes reset code
    # Assigns:    nothing
    # Returns:    nothing
    local __m   ; # Raw message to format
    local __l   ; # Line length available
    local __f1  ; # First line format
    local __f2  ; # Secondary line format
    local __s   ; # Optional suffix for last line
    local __c   ; # Optional foreground color code
    local __msg ; # Formatted message to print
    case $# in
        (0)
            __m=""
            __l=$MAXCOL
            __f1="\n%s"
            __f2="\n%s"
            __s=""
            __c=""
            ;;
        (1)
            __m="$1"
            __l=$MAXCOL
            __f1="\n%s"
            __f2="\n%s"
            __s=""
            __c=""
            ;;
        (2)
            __m="$1"
            __l=$2
            __f1="\n%s"
            __f2="\n%s"
            __s=""
            __c=""
            ;;
        (3)
            __m="$1"
            __l=$2
            __f1="$3"
            __f2="$3"
            __s=""
            __c=""
            ;;
        (4)
            __m="$1"
            __l=$2
            __f1="$3"
            __f2="$4"
            __s=""
            __c=""
            ;;
        (5)
            __m="$1"
            __l=$2
            __f1="$3"
            __f2="$4"
            __s="$5"
            __c=""
            ;;
        (6)
            __m="$1"
            __l=$2
            __f1="$3"
            __f2="$4"
            __s="$5"
            __c="$6"
            ;;
        (*)
            __m="$1"
            __l=$2
            __f1="$3"
            __f2="$4"
            __s="$5"
            __c="$6"
            ;;
    esac
    # Ensure __l is a positive integer
    __l=${__l##*[!0-9]*}
    __l=${__l:-$MAXCOL}
    # Use of awk's pattern matching is not optimal since gensub is not portable
    # and cannot be used like in printf(f,gensub(/^ /,"",1,substr($1,i,sl))); 
    __msg=`echo "$__m" |\
           awk -v sl=$__l \
               -v f1="$__f1" \
               -v f2="$__f2" \
               -v s="$__s" \
               -F '#' '{ \
                         f = f1; \
                         i = 1; \
                         while ( length(substr($1,i,sl)) > 0 ) { \
                             subs = substr($1,i,sl); \
                             sub("^ ","",subs); \
                             printf(f,subs); \
                             f = f2; \
                             i = i + sl; \
                         } \
                         i = 1;
                         while ( length(substr(s,i,sl)) > 0 ) { \
                             subs = substr(s,i,sl); \
                             sub("^ ","",subs); \
                             printf(f,subs); \
                             i = i + sl; \
                         } \
                       }'`

    if [ $((lflag || log)) -eq 1 ]; then
        # Escape codes must be removed before writing to log file.
        local __s1=${__msg//$P1/}
        local __s2=${__s1//$P2/}
        local __s3=${__s2//$P3/}
        echo $echo_n "$__s3" $echo_c >> $lfile
    fi
    # Escape codes are used for the terminal output only.
    if [ -t 1 ]; then
        # Escape codes must be activated before writing to terminal.
        __msg="$__c$__msg$RESET"
        __msg=${__msg//$P4/\\$P4}
        echo $echo_n $echo_escape "$__msg" $echo_c
    else
        # Escape codes must be removed before writing to standard output.
        local __s1=${__msg//$P1/}
        local __s2=${__s1//$P2/}
        local __s3=${__s2//$P3/}
        echo $echo_n "$__msg" $echo_c
    fi
}

#
# Debug: Prints a debug message in one or more justified lines within margins.
#
Debug() {
    # Parameters:
    # 1: Message to print
    # Globals:
    # - MAXCOL:
    # - LMARGIN: 
    # - RMARGIN: 
    # - MARGIN:
    # - FG_DEBUG:
    # - dbg: debug flag set with SHELL_DEBUG environment variable
    # - dflag: debug flag set with -d option
    # Assigns:
    # - FGCOLOR: Foreground color
    # Returns:    nothing
    if [ $((dflag || dbg)) -ne 1 ]; then
        return
    fi
    local __p1 ; # Prefix for first line
    local __p2 ; # Prefix for secondary line
    local __s1 ; # Suffix for first line
    local __s2 ; # Suffix for secondary line
    local __f1 ; # Format for first line
    local __f2 ; # Format for secondary lines
    local __l  ; # Length of left margin, prefix, message, suffix, right margin 
    local __s  ; # Section closing suffix
    __p1="|##> DEBUG:   "
    __p2="|##>          " ; # Both p1 and p2 must be the same length
    __s1=" <##|"
    __s2=" <##|" ; # Both s1 and s2 must be the same length
    __l=$(($MAXCOL - $LMARGIN - $RMARGIN - ${#__p1} - ${#__s1}))
    __f1="\n${MARGIN}${__p1}%-${__l}s${__s1}"
    __f2="\n${MARGIN}${__p2}%-${__l}s${__s2}"
    __s=""
    eval FGCOLOR=\$$(echo $FG_DEBUG)
    PrintLine "$1" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Error: Prints an error message in one or more justified lines within margins.
#
Error() {
    # Parameters:
    # 1: Message to print
    # Globals:
    # - MAXCOL:
    # - LMARGIN: 
    # - RMARGIN: 
    # - MARGIN:
    # - BLINK:
    # - RESETBLINK:
    # - FG_ERROR:
    # Assigns:
    # - FGCOLOR: Foreground color
    # Returns:    nothing
    local __p1
    local __p2
    local __s1
    local __s2
    local __f1
    local __f2
    local __l
    local __s
    __p1="|*** ${BLINK}ERROR:${RESETBLINK}   " ; # Embedded attribute escape
    __p2="|***          "
    __s1=" ***|"
    __s2=" ***|"
    __l=$(($MAXCOL - $LMARGIN - $RMARGIN - ${#__p2} - ${#__s1}))
    __f1="\n${MARGIN}${__p1}%-${__l}s${__s1}"
    __f2="\n${MARGIN}${__p2}%-${__l}s${__s2}"
    __s=""
    eval FGCOLOR=\$$(echo $FG_ERROR)
    PrintLine "$1" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Warning: Prints a warning message in one or more justified lines within
#          margins.
#
Warning() {
    # Parameters:
    # 1: Message to print
    # Globals:
    # - MAXCOL:
    # - LMARGIN: 
    # - RMARGIN: 
    # - MARGIN:
    # - BLINK:
    # - RESETBLINK:
    # - FG_WARN:
    # Assigns:
    # - FGCOLOR: Foreground color
    # Returns:    nothing
    local __p1
    local __p2
    local __s1
    local __s2
    local __f1
    local __f2
    local __l
    local __s
    __p1="|*** ${BLINK}WARNING:${RESETBLINK} " ; # Embedded attribute escape
    __p2="|***          "
    __s1=" ***|"
    __s2=" ***|"
    __l=$(($MAXCOL - $LMARGIN - $RMARGIN - ${#__p2} - ${#__s1}))
    __f1="\n${MARGIN}${__p1}%-${__l}s${__s1}"
    __f2="\n${MARGIN}${__p2}%-${__l}s${__s2}"
    __s=""
    eval FGCOLOR=\$$(echo $FG_WARN)
    PrintLine "$1" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Explain: Prints a continuation message in one or more justified lines within
#          margins.
#          Used to explain details of Error and Warning.
#
Explain() {
    # Parameters:
    # 1: Message to print
    # Globals:
    # - MAXCOL:
    # - LMARGIN: 
    # - RMARGIN: 
    # - MARGIN:
    # - FGCOLOR:
    # Assigns:    nothing
    # Returns:    nothing
    local __p1
    local __p2
    local __s1
    local __s2
    local __f1
    local __f2
    local __l
    local __s
    __p1="|***          "
    __p2="|***          "
    __s1=" ***|"
    __s2=" ***|"
    __l=$(($MAXCOL - $LMARGIN - $RMARGIN - ${#__p1} - ${#__s1}))
    __f1="\n${MARGIN}${__p1}%-${__l}s${__s1}"
    __f2="\n${MARGIN}${__p2}%-${__l}s${__s2}"
    __s=""
    PrintLine "$1" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Doing: Prints an opening message for a task section at the desired level
#        of indentation, in one or more justified lines within margins.
#
Doing() {
    # Parameters:
    # 1: Message to print
    # 2: Indentation level after the margin
    # Globals:
    # - MAXCOL:
    # - RMARGIN: 
    # - MARGIN:
    # - BOLD:
    # - RESET:
    # Assigns:
    # - FGCOLOR: Foreground color
    # Returns:    nothing
    local __p1
    local __p2
    local __s1
    local __s2
    local __f1
    local __f2
    local __l
    local __s
    case $2 in
        0)
           __p1="===> " ; # Make this string fit exactly in the length of MARGIN.
           __p2="${MARGIN}"
           __s1=""
           __s2=""
           ;;
        1)
           __p1="${MARGIN}---> "
           __p2="${MARGIN}     "
           __s1=""
           __s2=""
           ;;
        2)
           __p1="${MARGIN}     +++> "
           __p2="${MARGIN}          "
           __s1=""
           __s2=""
           ;;
        3)
           __p1="${MARGIN}          ###> "
           __p2="${MARGIN}               "
           __s1=""
           __s2=""
           ;;
       (*)
           __p1="===> "
           __p2="${MARGIN}"
           __s1=""
           __s2=""
           ;;
    esac
    __l=$(($MAXCOL - $RMARGIN - ${#__p1} - ${#__s1}))
    __f1="\n${BOLD}${__p1}${RESET}%-${__l}s${__s1}"
    __f2="\n${__p2}%-${__l}s${__s2}"
    __s=""
    FGCOLOR=$RESET
    PrintLine "$1" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Done: Prints a closing message for a task section at the desired level
#       of indentation, in one or more justified lines within margins.
#
Done() {
    # Parameters:
    # 1: Message to print
    # 2: Indentation level after the margin
    # Globals:
    # - MAXCOL:
    # - LMARGIN: 
    # - RMARGIN: 
    # - MARGIN:
    # - BOLD:
    # - RESET:
    # Assigns:
    # - FGCOLOR: Foreground color
    # Returns:    nothing
    local __p1
    local __p2
    local __s1
    local __s2
    local __f1
    local __f2
    local __l
    local __s
    case $2 in
        0)
           __p1="<=== Done "
           __p2="${MARGIN}"
           __s1=""
           __s2=""
           # __s="<==="
           ;;
        1)
           __p1="     <--- Done "
           __p2="${MARGIN}     "
           __s1=""
           __s2=""
           # __s="<---"
           ;;
        2)
           __p1="          <+++ Done "
           __p2="${MARGIN}          "
           __s1=""
           __s2=""
           # __s="<+++"
           ;;
        3)
           __p1="               <### Done "
           __p2="${MARGIN}               "
           __s1=""
           __s2=""
           # __s="<###"
           ;;
       (*)
           __p1="<=== Done "
           __p2="${MARGIN}"
           __s1=""
           __s2=""
           # __s="<==="
           ;;
    esac
    __l=$(($MAXCOL - $LMARGIN - $RMARGIN - ${#__p1} - ${#__s1}))
    __f1="\n${MARGIN}${BOLD}${__p1}${RESET}%-${__l}s${__s1}"
    __f2="\n${MARGIN}${__p2}%-${__l}s${__s2}"
    # __s="${BOLD}$__s${RESET}"
    __s=""
    FGCOLOR=$RESET
    PrintLine "$1" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Tell: Prints an intermediate message in a task section at the desired level
#       of indentation, in one or more justified lines within margins.
#
Tell() {
    # Parameters:
    # 1: Message to print
    # 2: Indentation level after the margin
    # Globals:
    # - MAXCOL:
    # - LMARGIN: 
    # - RMARGIN: 
    # - MARGIN:
    # - RESET:
    # Assigns:
    # - FGCOLOR: Foreground color
    # Returns:    nothing
    local __p1
    local __p2
    local __s1
    local __s2
    local __f1
    local __f2
    local __l
    local __s
    case $2 in
        0)
           __p1=""
           __p2=""
           __s1=""
           __s2=""
           ;;
        1)
           __p1="     "
           __p2="     "
           __s1=""
           __s2=""
           ;;
        2)
           __p1="          "
           __p2="          "
           __s1=""
           __s2=""
           ;;
        3)
           __p1="               "
           __p2="               "
           __s1=""
           __s2=""
           ;;
       (*)
           __p1=""
           __p2=""
           __s1=""
           __s2=""
           ;;
    esac
    __l=$(($MAXCOL - $LMARGIN - $RMARGIN - ${#__p1} - ${#__s1}))
    __f1="\n${MARGIN}${__p1}%-${__l}s${__s1}"
    __f2="\n${MARGIN}${__p2}%-${__l}s${__s2}"
    __s=""
    FGCOLOR=$RESET
    PrintLine "$1" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Banner: Prints a message in one or more justified lines within margins.
#
Banner() {
    # Parameters:
    # 1: Message to print
    # 2: Indentation level after the margin
    # 3: Optional text foreground color name
    # Globals:
    # - MAXCOL:
    # - LMARGIN: 
    # - RMARGIN: 
    # - MARGIN:
    # - RESET:
    # Assigns:
    # - FGCOLOR: Foreground color
    # Returns:    nothing
    local __fgc=${3:-"RESET"}
    local __p1
    local __p2
    local __s1
    local __s2
    local __f1
    local __f2
    local __l
    local __s
    case $2 in
        0)
           __p1="| "
           __p2="| "
           __s1=" |"
           __s2=" |"
           ;;
        1)
           __p1="|       "
           __p2="|       "
           __s1=" |"
           __s2=" |"
           ;;
        2)
           __p1="|            "
           __p2="|            "
           __s1=" |"
           __s2=" |"
           ;;
        3)
           __p1="|                "
           __p2="|                "
           __s1=" |"
           __s2=" |"
           ;;
       (*)
           __p1="| "
           __p2="| "
           __s1=" |"
           __s2=" |"
           ;;
    esac
    __l=$(($MAXCOL - $LMARGIN - $RMARGIN - ${#__p1} - ${#__s1}))
    __f1="\n${MARGIN}${__p1}%-${__l}s${__s1}"
    __f2="\n${MARGIN}${__p2}%-${__l}s${__s2}"
    __s=""
    reply=$(CheckRange $__fgc E "$FGC_LIST RESET")
    test $reply -ne 0 && __fgc=CURRENT
    test "$__fgc" != "CURRENT" && eval FGCOLOR=\$$(echo $__fgc)
    PrintLine "$1" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Filler: Prints an empty justified line within margins.
#
Filler() {
    # Parameters: none
    # Globals:
    # - MAXCOL:
    # - LMARGIN: 
    # - RMARGIN: 
    # - MARGIN:
    # - HIDDEN:
    # - RESETHIDDEN:
    # - RESET:
    # Assigns:
    # - FGCOLOR: Foreground color
    # Returns:    nothing
    local __p1
    local __p2
    local __s1
    local __s2
    local __f1
    local __f2
    local __l
    local __s
    __p1="| "
    __p2="| "
    __s1=" |"
    __s2=" |"
    __l=$(($MAXCOL - $LMARGIN - $RMARGIN - ${#__p1} - ${#__s1}))
    __f1="\n${MARGIN}${__p1}${HIDDEN}%-${__l}s${RESETHIDDEN}${__s1}"
    __f2="\n${MARGIN}${__p2}%-${__l}s${__s2}"
    __s=""
    # FGCOLOR=$RESET
    PrintLine "_" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Frame: Prints a line of replicated characters as long as margins permit.
#
Frame() {
    # Parameters:
    # 1: Optional text foreground color name
    # Globals:
    # - MAXCOL:
    # - LMARGIN: 
    # - RMARGIN: 
    # - MARGIN:
    # - RESET:
    # Assigns:
    # - FGCOLOR: Foreground color
    # Returns:    nothing
    local __fgc=${1:-"RESET"}
    local __p1
    local __p2
    local __s1
    local __s2
    local __f1
    local __f2
    local __l
    local __s
    local __frm
    local __str
    __p1="+"
    __p2=""
    __s1="+"
    __s2=""
    __l=$(($MAXCOL - $LMARGIN - $RMARGIN - ${#__p1} - ${#__s1}))
    __f1="\n${MARGIN}${__p1}%-${__l}s${__s1}"
    __f2="%s"
    __s=""
    __str=$(printf "%${__l}s")
    __frm=${__str// /-} ; # Here is the character that forms the line
    reply=$(CheckRange $__fgc E "$FGC_LIST RESET")
    test $reply -ne 0 && __fgc=CURRENT
    test "$__fgc" != "CURRENT" && eval FGCOLOR=\$$(echo $__fgc)
    PrintLine "$__frm" $__l "$__f1" "$__f2" "$__s" "$FGCOLOR"
}

#
# Standard Library END
######################

###############################
# Standard Initialization BEGIN
#
#********************#
# Harcoded Constants #
#********************#
#
# Date and time formats
DF='%Y-%m-%d' ; # Date format YYYY-MM-DD
TF='%H:%M:%S' ; # Time format HH:MM:SS
YF='%Y'       ; # Year format YYYY

# Integer arithmetics
declare -r MININT32=-2147483648 ; # Minimum integer value
declare -r MAXINT32=2147483647  ; # Maximum integer value

# Text formatting for display and log file
LIFS=';'  ; # Local Internal Field Separator
LSEP='='  ; # Local Separator
LMARGIN=5 ; # Number of columns on the left margin
RMARGIN=5 ; # Number of columns in the right margin
MINLIN=70 ; # Minimum number of lines in the terminal output
MINCOL=80 ; # Maximum number of lines in the terminal output

# Foreground codes.
# - Must be the name of one of the escape codes for 'echo',
#   listed in derived constant FGC_LIST.
FG_WARN=YELLOW
FG_ERROR=RED
FG_DEBUG=CYAN

# Exit codes
EC_END=0    ; # Normal end of program with message to output device
EC_CANCEL=1 ; # Cancelled by user (^C)
EC_ABORT=2  ; # Aborted by error
EC_REJECT=3 ; # Rejected by verification
EC_INFO=4   ; # Executed only for informative purposes (syntax, usage, version)
EC_QUIET=9  ; # End without any message to output device

# BASH Versions supported
SHELL_OLDEST_VERSION="2.03"

#*******************#
# System Dependency #
#*******************#
#
# Check BASH version compatibility
#
ShellVersion
if [ $? -ne 0 ]; then
    exit $EC_CANCEL
fi

#
# System flavor and portability
#
UNAME=`uname -s`
OSREL=`uname -r`
OSTAG="$(SystemType "$UNAME" "$OSREL" "${HOST:-`hostname`}")"
Portable "$OSTAG" "$OSREL"

#
# Architecture detection
#
# Register width
CPUHW="$(Architecture `uname $uname_machine`)"
BITS=`getconf LONG_BIT 2>&-`
if [ $? -ne 0 ]; then
    BITS=`echo $CPUHW | cut -d'_' -f2`
fi
# Endianess
endian=`echo I | tr -d [:space:] | od -to2 | head -n1 | awk '{print $2}' | cut -c6`
if [ $endian -ne 0 ]; then
    BIGENDIAN=0
else
    BIGENDIAN=1
fi

#*********************#
# Data Initialization #
#*********************#
#
# Terminal configuration
#
save_tty=`stty -g`

#
# Program Name, and Path
#
progname=`basename $0`
progpath=`dirname $0` 

if [ "$progpath" = "." ]; then
    progpath=`pwd`
fi

#
# Date and Time Stamps
#
datestamp=`date +$DF`
yearstamp=`date +$YF`
timestamp=`date +$TF`

#
# Derived Constants
#
# Text formatting for display and log file
# - resize has a bug that causes it to return an empty string.
#   To reproduce it, execute 
#   ~> sleep 1; r=`resize`; echo ">$r<"  <ENTER> <ENTER>
#
#   resize: unknown character, exiting.
#   ~> 
#   Note that you need to press ENTER twice.
#   The normal result when ENTER is pressed just once, is:
#   ~> sleep 1; r=`resize`; echo ">$r<"  <ENTER>
#   >COLUMNS=120;
#   LINES=40;
#   export COLUMNS LINES;<
#   ~> 
#
# LINES=`resize 2>&- | awk '/LINES=/{i=split($1,a,"="); printf("%d\n",a[i])}'`
# COLUMNS=`resize 2>&- | awk '/COLUMNS=/{i=split($1,a,"="); printf("%d\n",a[i])}'`
#
# The incidence of the bug can be minimized using stty, and checking
# string contents with a fallback.
LINCOL=`stty $stty_size`
LINCOL=${LINCOL:-"$MINLIN $MINCOL"}
LINES=`echo $LINCOL | awk '{printf("%d",$1)}'`
COLUMNS=`echo $LINCOL | awk '{printf("%d",$2)}'`
MAXCOL=${COLUMNS:-$MINCOL} ; # Max number of columns in the terminal output
MAXLIN=${LINES:-$MINLIN} ; # Max number of lines in the terminal output
MARGIN=$(printf "%*s" ${LMARGIN}) ; # String of blanks for left margin
SEPARATOR=$(printf "%*s" ${MAXCOL})
SEPARATOR=${SEPARATOR// /$LSEP} ; # Here is the character that forms the line

# Log file path
LOGFILE="${progpath}/${progname%.*}.log" ; # '.*' matches the shortest substring
                                           # beginning with '.', and % strips
                                           # it from end.

# Escape codes for 'echo'
TAT_LIST=""
FGC_LIST=""
if [ $(tput colors) -gt 0 ] ; then
    # - Light foreground colors on current background
    #   30..38
    FGC_LIST="\
              BLACK \
              RED \
              GREEN \
              YELLOW \
              BLUE \
              MAGENTA \
              CYAN \
              GRAY \
              OLIVE \
             "
    j=30
    for c in $FGC_LIST ; do
        # A variable named as the foreground color is created and
        # assigned with the corresponding escape code.
        declare $c="${e}[0;${j}m"
        # echo "$c = >$(eval echo \$$c)<"
        j=$((j + 1))
    done

    # - Text attributes
    declare -a pair
    for p in \
             "BOLD 1"\
             "BLINK 5"\
             "HIDDEN 8"\
             "REVERSE 7"\
             "UNDERLINED 4"\
             "RESET 0"\
             "RESETBOLD 21"\
             "RESETBLINK 25"\
             "RESETHIDDEN 28"\
             "RESETREVERSE 27"\
             "RESETUNDERLINE 24"\
             ; do
        pair=($p)
        TAT_LIST="${TAT_LIST}${pair[0]} "
        # A variable named as the text attribute is created and
        # assigned with the corresponding escape code.
        declare ${pair[0]}="${e}[${pair[1]}m"
        # echo "${pair[0]} = >$(eval echo \$${pair[0]})<"
    done
fi

# Patterns to match the echo escape codes for text attributes and colors.
P1=${e}'[?m'
P2=${e}'[??m'
P3=${e}'[0;??m'
P4=${e}'['

# Help lines
HELP_LINE="\
 [[-d|--debug] [-l|--log]$LIFS\
  [--help] [--usage]$LIFS\
  [--License] [--Version]$LIFS\
  --]$LIFS\
          "

USAGE_LINE="\
 [[-d|--debug] $LIFS\
  [-l|--log] $LIFS\
  [--help] $LIFS\
  [--usage] $LIFS\
  [--License] $LIFS\
  [--Version] $LIFS\
  --] $LIFS\
 where: $LIFS\
   -d or --debug: enables debugging mode $LIFS\
   -l or --log: enables log to file $LIFS\
   -m or --max: maximum number of records to display $LIFS\
  --help: displays command line syntax $LIFS\
  --usage: displays this Usage Screen $LIFS\
  --License: displays copyright and license $LIFS\
  --Version: displays the program version $LIFS\
  --: marks the end of options, starts positional arguments $LIFS\
          "

LICENSE_LINE="\
  Copyright (C) $(ExpandRange 2013..$yearstamp)$LIFS\
  Pedro Baksai (pbaksai@eso.org)$LIFS\
  $LIFS\
  This program is free software: you can redistribute$LIFS\
  it and/or modify it under the terms of the$LIFS\
  GNU Lesser General Public License as published by$LIFS\
  the Free Software Foundation, either version 3 of$LIFS\
  the License, or (at your option) any later version.$LIFS\
  $LIFS\
  This program is distributed in the hope that it$LIFS\
  will be useful, but WITHOUT ANY WARRANTY,$LIFS\
  without even the implied warranty of$LIFS\
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.$LIFS\
  See the GNU General Public License for more details.$LIFS\
  $LIFS\
  You should have received a copy of the$LIFS\
  GNU General Public License along with this program.$LIFS\
  If not, see <http://www.gnu.org/licenses/lgpl.html>.$LIFS\
             "

#
# Environment variables
#
user=${USER:-`whoami`}
ws=${HOST:-`hostname`}
hd=${HOME:-`pwd`}
bv=${BASH_VERSION:-'?'}
log=`echo ${SHELL_LOG:-0} | tr '[:upper:]' '[:lower:]'`
dbg=`echo ${SHELL_DEBUG:-0} | tr '[:upper:]' '[:lower:]'`

#
# Global variables
#
if [ "${log}" = ""    -o \
     "${log}" = "0"   -o \
     "${log}" = "off" -o \
     "${log}" = "no" ]; then
    log=0
else
    log=1
fi

if [ "${dbg}" = ""    -o \
     "${dbg}" = "0"   -o \
     "${dbg}" = "off" -o \
     "${dbg}" = "no" ]; then
    dbg=0
else
    dbg=1
fi

# Debug, error, log flags
dflag=0
eflag=0
lflag=0

# Current log file
lfile="$LOGFILE"

# Current text foreground color
FGCOLOR=""

# Array of process id
declare -a PID
PID[0]=$$
p=0

# Array of external variables
declare -a VAR
v=0

# Array of ordered literals
declare -a LIT
l=0
for i in {0,1,2,3,4,5,6,7,8,9,a,b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,x,y,z,A,B,C,D,E,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z}; do
    LIT[$l]=$i
    l=$((l + 1))
done

#***********************#
# System Initialization #
#***********************#
#
# Signals handling
#
SignalHandlers

#
# Log file
#
# StartLog

#
# Standard Initialization END
#############################

